/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * amd.c
 *
 * AMD: AMUSIC Adlib Tracker player
 */

#include <string.h>

#include "../player.h"
#include "../tracker.h"
#include "amd.h"

#define GET_INT8(addr)  (*addr++)
#define GET_INT16(addr) (*addr++) + 256 * (*addr++)

static const uint8_t g_amd_id_1[9]  = { 0x3C, 0x6F, 0xEF, 0x51, 0x55, 0xEE, 0x52, 0x6F, 0x52 };
static const uint8_t g_amd_id_2[10] = { "MaDoKaN96" };

static char g_song_name[24];
static char g_author[24];

bool load(const char *file_name)
{
    uint8_t i, j;

    g_roboplay_interface->open(file_name, false);
    g_roboplay_interface->read((void *)READ_BUFFER, AMD_TOTAL_HEADER_SIZE);

    AMD_HEADER *amd_header = (AMD_HEADER *)READ_BUFFER; 

    bool found_amd_id_1 = true;
    bool found_amd_id_2 = true;
    for(i = 0; i < 9; i++)
    {
        if(amd_header->id[i] != g_amd_id_1[i]) found_amd_id_1 = false;
        if(amd_header->id[i] != g_amd_id_2[i]) found_amd_id_2 = false;
    }

    if(!found_amd_id_1 && !found_amd_id_2)
    {
        g_roboplay_interface->close();
        return false;        
    }

    for(i = 0; i < 24; i++)
    {
        g_song_name[i] = amd_header->song_name[i];
        g_author[i]    = amd_header->author[i];
    }

    TRACKER_INSTRUMENT instrument;
    instrument.arpeggio_start = 0;
    instrument.arpeggio_speed = 0;
    instrument.arpeggio_position = 0;
    instrument.arpeggio_speed_count = 0;
    instrument.misc = 0;
    instrument.slide = 0;
    for(i = 0; i < 26; i++)
    {
        for(j = 0; j < 11; j++) instrument.data[j] = amd_header->instruments[i].data[j];
        add_instrument(i, &instrument);
    }

    if(amd_header->version == 0x10)
    {
        /* Unpacked module */
    }
    else
    {
        /* Packed module */
        uint8_t *track_data = (uint8_t *)amd_header->track_data;
        for(i = 0; i < amd_header->number_of_patterns; i++)
            for(j = 0; j < 9; j++)
                g_track_order[i][j] = GET_INT16(track_data);

        uint16_t number_of_tracks = GET_INT16(track_data);
        uint16_t max_track_number = 0;
        for(uint16_t k = 0; k < number_of_tracks; k++)
        {
            uint16_t track_number = GET_INT16(track_data);
            if(track_number > 575) track_number = 575;
            max_track_number = ((track_number + 1) > max_track_number) ? (track_number + 1) : max_track_number;

            add_track(track_number);

            TRACKER_TRACK_DATA track;
            j = 0;
            do
            {
                uint8_t buffer = GET_INT8(track_data);
                if(buffer & 128)
                {
                    track.command = 0;
                    track.instrument = 0;
                    track.note = 0;
                    track.parameter_1 = 0;
                    track.parameter_2 = 0;

                    for(uint8_t t = j; t < j + (buffer & 127) && t < 64; t++)
                        add_track_data(&track);

                    j += buffer & 127;
                    continue;
                }

                track.parameter_2 = buffer % 10;
                track.parameter_1 = buffer / 10;
                buffer = GET_INT8(track_data);
                track.instrument = buffer >> 4;
                track.command = buffer & 0x0F;
                buffer = GET_INT8(track_data);
                if(buffer >> 4)
                    track.note = ((buffer & 14) >> 1) * 12 + (buffer >> 4);
                else
                    track.note = 0;
                track.instrument += (buffer & 1) << 4;
                add_track_data(&track);

                j++;
            } while(j < 64);
        }
    }

    g_roboplay_interface->close();
    
    return true;
}

float get_refresh()
{
    return 50.0;
}

uint8_t get_subsongs()
{
    return 0;
}

char* get_player_info()
{
    return "AMUSIC Adlib Tracker player by RoboSoft Inc.";
}

char* get_title()
{
    return g_song_name;
}

char* get_author()
{
    return g_author;
}

char* get_description()
{
    return "-";
}
