/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * mfm.h
 *
 * MFM: MoonBlaster FM
 */

#ifndef __MFM_H
#define __MFM_H

#include <stdint.h>
#include <stdbool.h>

#define NR_WAVE_CHANNELS 6
#define NR_WAVES         32

#define MAX_OWN_TONES    64
#define MAX_OWN_PATCHES  48

#define MAX_FM_CHANNELS   18
#define MAX_4OP_CHANNELS  6

#define NR_OF_2OP_INSTRUMENTS 24
#define NR_OF_4OP_INSTRUMENTS 12
#define NR_OF_VOICE_BANKS     36

#define INSTRUMENT_SIZE_2OP 11
#define INSTRUMENT_SIZE_4OP 22

#define COMMAND_CHANNEL  24

#define MAX_SONG_SEGMENTS 3

#define MAX_PATTERN  79
#define MAX_POSITION 219
#define MAX_STEP     15

#define STEP_BUFFER_SIZE 25

#define PATTERNSIZE STEP_BUFFER_SIZE * (MAX_STEP + 1)

#define NR_MODULATION_WAVES 3
#define MODULATION_WAVE_LENGTH 16

#define SONG_NAME_LENGTH     50
#define WAVE_KIT_NAME_LENGTH 8

#define FM_NOTE_ON           96      /* 001 - 096 */
#define FM_NOTE_OFF          97      /* 097       */
#define FM_INSTRUMENT_CHANGE 98      /* 098 - 121 */
#define FM_VOLUME_CHANGE     122     /* 122 - 185 */
#define FM_STEREO_SET        186     /* 186 - 188 */
#define FM_NOTE_LINK         189     /* 189 - 207 */
#define FM_PITCH_BEND        208     /* 208 - 226 */
#define FM_BRIGHTNESS        227     /* 227 - 239 */
#define FM_DETUNE            240     /* 240 - 246 */
#define FM_MODULATION        247     /* 247 - 249 */
#define FM_NONE              250

#define WAVE_NOTE_ON         96      /* 001 - 096 */
#define WAVE_NOTE_OFF        97      /* 097       */
#define WAVE_CHANGE          98      /* 098 - 145 */
#define WAVE_VOLUME_CHANGE   146     /* 146 - 177 */
#define WAVE_STEREO_SET      178     /* 178 - 192 */
#define WAVE_NOTE_LINK       193     /* 193 - 211 */
#define WAVE_PITCH_BEND      212     /* 212 - 230 */
#define WAVE_DETUNE          231     /* 231 - 237 */
#define WAVE_MODULATION      238     /* 238 - 240 */
#define WAVE_DAMP            241     /* 241 - 242 */
#define WAVE_NONE            243

#define COMMAND_TEMPO          23   /* 001 - 023 */
#define COMMAND_PATTERN_END    24   /* 024       */
#define COMMAND_STATUS_BYTE    25   /* 025 - 027 */
#define COMMAND_TRANSPOSE      28   /* 028 - 075 */
#define COMMAND_NONE           76

#define FREQUENCY_MODE_NORMAL     0
#define FREQUENCY_MODE_PITCH_BEND 1
#define FREQUENCY_MODE_MODULATION 2

#define REVERB_ENABLED  0x08
#define REVERB_DISABLED 0x00

typedef struct
{
    uint8_t song_length;
    uint8_t loop_position;
    uint8_t fm_voice_data_2op[NR_OF_2OP_INSTRUMENTS][INSTRUMENT_SIZE_2OP];
    uint8_t fm_voice_data_4op[NR_OF_4OP_INSTRUMENTS][INSTRUMENT_SIZE_4OP];
    uint8_t fm_stereo[MAX_FM_CHANNELS];
    uint8_t stereo[NR_WAVE_CHANNELS];
    uint8_t tempo;
    bool    hz_equalizer;
    uint8_t modulation_depth;
    int8_t  fm_detune[MAX_FM_CHANNELS];
    int8_t  wave_detune[NR_WAVE_CHANNELS];
    uint8_t nr_of_4op_channels;
    int8_t  modulation[NR_MODULATION_WAVES][MODULATION_WAVE_LENGTH];
    uint8_t fm_start_voices[MAX_FM_CHANNELS];
    uint8_t start_waves[NR_WAVE_CHANNELS];
    uint8_t wave_numbers[NR_WAVES];
    uint8_t wave_volumes[NR_WAVES];
    char    song_name[SONG_NAME_LENGTH];
    uint8_t fm_voice_banks[NR_OF_VOICE_BANKS];
    char    wave_kit_name[WAVE_KIT_NAME_LENGTH];
} MFM_HEADER;

typedef struct
{
    uint16_t size;
    uint8_t  nr_of_patterns;
} MFM_PATTERN_HEADER;

typedef struct
{
    uint8_t   last_note;
    uint8_t   frequency_mode;
    int16_t   pitch_bend_speed;
    uint8_t   modulation_index;
    uint8_t   modulation_count;
    int8_t    detune_value;
    uint8_t   last_instrument;
    uint8_t   next_frequency_low;
    uint8_t   next_frequency_high;
    uint8_t   last_frequency_low;
    uint8_t   last_frequency_high;
} MFM_FM_STATUS_TABLE;

typedef struct 
{
    uint8_t   last_note;
    uint8_t   frequency_mode;
    int16_t   pitch_bend_speed;
    uint8_t   modulation_index;
    uint8_t   modulation_count;
    int8_t    detune_value;
    uint8_t   next_tone_low;
    uint8_t   next_tone_high;
    uint16_t  next_frequency;
    uint8_t   current_wave;
    int8_t    current_stereo;
    uint8_t   pseudo_reverb;
    uint8_t  *header_bytes;
    uint8_t   volume;
    uint16_t *frequency_table;
    uint16_t  pitch_frequency;
} MFM_WAVE_STATUS_TABLE;

typedef struct 
{
    bool    second_opl_port;
    uint8_t frequency_register;
    uint8_t volume_register;   
} MFM_FM_PORT_DATA;

static const uint8_t g_4op_registers[MAX_4OP_CHANNELS][INSTRUMENT_SIZE_4OP] =
{
    { 0x20, 0x23, 0x40, 0x43, 0x60, 0x63, 0x80, 0x83, 0xE0, 0xE3, 0x28, 0x2B, 0x48, 0x4B, 0x68, 0x6B, 0x88, 0x8B, 0xE8, 0xEB, 0xC0, 0xC3 },
    { 0x21, 0x24, 0x41, 0x44, 0x61, 0x64, 0x81, 0x84, 0xE1, 0xE4, 0x29, 0x2C, 0x49, 0x4C, 0x69, 0x6C, 0x89, 0x8C, 0xE9, 0xEC, 0xC1, 0xC4 },
    { 0x22, 0x25, 0x42, 0x45, 0x62, 0x65, 0x82, 0x85, 0xE2, 0xE5, 0x2A, 0x2D, 0x4A, 0x4D, 0x6A, 0x6D, 0x8A, 0x8D, 0xEA, 0xED, 0xC2, 0xC5 },
    { 0x20, 0x23, 0x40, 0x43, 0x60, 0x63, 0x80, 0x83, 0xE0, 0xE3, 0x28, 0x2B, 0x48, 0x4B, 0x68, 0x6B, 0x88, 0x8B, 0xE8, 0xEB, 0xC0, 0xC3 },
    { 0x21, 0x24, 0x41, 0x44, 0x61, 0x64, 0x81, 0x84, 0xE1, 0xE4, 0x29, 0x2C, 0x49, 0x4C, 0x69, 0x6C, 0x89, 0x8C, 0xE9, 0xEC, 0xC1, 0xC4 },
    { 0x22, 0x25, 0x42, 0x45, 0x62, 0x65, 0x82, 0x85, 0xE2, 0xE5, 0x2A, 0x2D, 0x4A, 0x4D, 0x6A, 0x6D, 0x8A, 0x8D, 0xEA, 0xED, 0xC2, 0xC5 }
};

static const uint8_t g_2op_registers[MAX_FM_CHANNELS][INSTRUMENT_SIZE_2OP] = 
{
    { 0x32, 0xF2, 0xF5, 0xC8 },
    { 0x31, 0xF1, 0xF4, 0xC7 },
    { 0x30, 0xF0, 0xF3, 0xC6 },
    { 0x32, 0xF2, 0xF5, 0xC8 },
    { 0x31, 0xF1, 0xF4, 0xC7 },
    { 0x30, 0xF0, 0xF3, 0xC6 },

    { 0x2A, 0xEA, 0xED, 0xC5 },
    { 0x22, 0xE2, 0xE5, 0xC2 },
    { 0x29, 0xE9, 0xEC, 0xC4 },
    { 0x21, 0xE1, 0xE4, 0xC1 },
    { 0x28, 0xE8, 0xEB, 0xC3 },
    { 0x20, 0xE0, 0xE3, 0xC0 },

    { 0x2A, 0xEA, 0xED, 0xC5 },
    { 0x22, 0xE2, 0xE5, 0xC2 },
    { 0x29, 0xE9, 0xEC, 0xC4 },
    { 0x21, 0xE1, 0xE4, 0xC1 },
    { 0x28, 0xE8, 0xEB, 0xC3 },
    { 0x20, 0xE0, 0xE3, 0xC0 }
};

static const MFM_FM_PORT_DATA g_port_data_4op[MAX_4OP_CHANNELS] =
{
    { false, 0xA0, 0x43 },
    { false, 0xA1, 0x44 },
    { false, 0xA2, 0x45 },
    { true,  0xA0, 0x43 },
    { true,  0xA1, 0x44 },
    { true,  0xA2, 0x45 },
};

static const MFM_FM_PORT_DATA g_port_data_2op[MAX_FM_CHANNELS] =
{
    { true,  0xA8, 0x55 },
    { true,  0xA7, 0x54 },
    { true,  0xA6, 0x53 },
    { false, 0xA8, 0x55 },
    { false, 0xA7, 0x54 },
    { false, 0xA6, 0x53 },

    { true,  0xA5, 0x4D },
    { true,  0xA2, 0x45 },
    { true,  0xA4, 0x4C },
    { true,  0xA1, 0x44 },
    { true,  0xA3, 0x4B },
    { true,  0xA0, 0x43 },

    { false, 0xA5, 0x4D },
    { false, 0xA2, 0x45 },
    { false, 0xA4, 0x4C },
    { false, 0xA1, 0x44 },
    { false, 0xA3, 0x4B },
    { false, 0xA0, 0x43 }
};

static const uint16_t g_frequency_table[] =
{
    /*   
        C 1       C#1     D 1      D#1       E 1       F 1       
        F#1       G 1     G#1      A 1       A#1       B 1 
    */
    0x015A, 0x016E, 0x0184, 0x019B, 0x01B3, 0x01CD,
    0x01E9, 0x0206, 0x0224, 0x0245, 0x0268, 0x028C,
    0x055A, 0x056E, 0x0584, 0x059B, 0x05B3, 0x05CD,
    0x05E9, 0x0606, 0x0624, 0x0645, 0x0668, 0x068C,
    0x095A, 0x096E, 0x0984, 0x099B, 0x09B3, 0x09CD,
    0x09E9, 0x0A06, 0x0A24, 0x0A45, 0x0A68, 0x0A8C,
    0x0D5A, 0x0D6E, 0x0D84, 0x0D9B, 0x0DB3, 0x0DCD,
    0x0DE9, 0x0E06, 0x0E24, 0x0E45, 0x0E68, 0x0E8C,
    0x115A, 0x116E, 0x1184, 0x119B, 0x11B3, 0x11CD,
    0x11E9, 0x1206, 0x1224, 0x1245, 0x1268, 0x128C,
    0x155A, 0x156E, 0x1584, 0x159B, 0x15B3, 0x15CD,
    0x15E9, 0x1606, 0x1624, 0x1645, 0x1668, 0x168C,
    0x195A, 0x196E, 0x1984, 0x199B, 0x19B3, 0x19CD,
    0x19E9, 0x1A06, 0x1A24, 0x1A45, 0x1A68, 0x1A8C,
    0x1D5A, 0x1D6E, 0x1D84, 0x1D9B, 0x1DB3, 0x1DCD,
    0x1DE9, 0x1E06, 0x1E24, 0x1E45, 0x1E68, 0x1E8C
};

static const uint8_t g_table_4op_on[] =
{
    0b000001, 0b000011, 0b000111, 0b001111, 0b011111, 0b111111
};

static const uint8_t g_tabdiv12[][2] =
{
    {0xB0, 0x00}, {0xB0, 0x02}, {0xB0, 0x04}, {0xB0, 0x06}, {0xB0, 0x08}, {0xB0, 0x0A}, {0xB0, 0x0C}, {0xB0, 0x0E}, {0xB0, 0x10}, {0xB0, 0x12}, {0xB0, 0x14}, {0xB0, 0x16},
    {0xC0, 0x00}, {0xC0, 0x02}, {0xC0, 0x04}, {0xC0, 0x06}, {0xC0, 0x08}, {0xC0, 0x0A}, {0xC0, 0x0C}, {0xC0, 0x0E}, {0xC0, 0x10}, {0xC0, 0x12}, {0xC0, 0x14}, {0xC0, 0x16},
    {0xD0, 0x00}, {0xD0, 0x02}, {0xD0, 0x04}, {0xD0, 0x06}, {0xD0, 0x08}, {0xD0, 0x0A}, {0xD0, 0x0C}, {0xD0, 0x0E}, {0xD0, 0x10}, {0xD0, 0x12}, {0xD0, 0x14}, {0xD0, 0x16},
    {0xE0, 0x00}, {0xE0, 0x02}, {0xE0, 0x04}, {0xE0, 0x06}, {0xE0, 0x08}, {0xE0, 0x0A}, {0xE0, 0x0C}, {0xE0, 0x0E}, {0xE0, 0x10}, {0xE0, 0x12}, {0xE0, 0x14}, {0xE0, 0x16},
    {0xF0, 0x00}, {0xF0, 0x02}, {0xF0, 0x04}, {0xF0, 0x06}, {0xF0, 0x08}, {0xF0, 0x0A}, {0xF0, 0x0C}, {0xF0, 0x0E}, {0xF0, 0x10}, {0xF0, 0x12}, {0xF0, 0x14}, {0xF0, 0x16},
    {0x00, 0x00}, {0x00, 0x02}, {0x00, 0x04}, {0x00, 0x06}, {0x00, 0x08}, {0x00, 0x0A}, {0x00, 0x0C}, {0x00, 0x0E}, {0x00, 0x10}, {0x00, 0x12}, {0x00, 0x14}, {0x00, 0x16},
    {0x10, 0x00}, {0x10, 0x02}, {0x10, 0x04}, {0x10, 0x06}, {0x10, 0x08}, {0x10, 0x0A}, {0x10, 0x0C}, {0x10, 0x0E}, {0x10, 0x10}, {0x10, 0x12}, {0x10, 0x14}, {0x10, 0x16},
    {0x20, 0x00}, {0x20, 0x02}, {0x20, 0x04}, {0x20, 0x06}, {0x20, 0x08}, {0x20, 0x0A}, {0x20, 0x0C}, {0x20, 0x0E}, {0x20, 0x10}, {0x20, 0x12}, {0x20, 0x14}, {0x20, 0x16},
    {0x30, 0x00}, {0x30, 0x02}, {0x30, 0x04}, {0x30, 0x06}, {0x30, 0x08}, {0x30, 0x0A}, {0x30, 0x0C}, {0x30, 0x0E}, {0x30, 0x10}, {0x30, 0x12}, {0x30, 0x14}, {0x30, 0x16},
    {0x40, 0x00}, {0x40, 0x02}, {0x40, 0x04}, {0x40, 0x06}, {0x40, 0x08}, {0x40, 0x0A}, {0x40, 0x0C}, {0x40, 0x0E}, {0x40, 0x10}, {0x40, 0x12}, {0x40, 0x14}, {0x40, 0x16},
    {0x50, 0x00}, {0x50, 0x02}, {0x50, 0x04}, {0x50, 0x06}, {0x50, 0x08}, {0x50, 0x0A}, {0x50, 0x0C}, {0x50, 0x0E}, {0x50, 0x10}, {0x50, 0x12}, {0x50, 0x14}, {0x50, 0x16},
    {0x60, 0x00}, {0x60, 0x02}, {0x60, 0x04}, {0x60, 0x06}, {0x60, 0x08}, {0x60, 0x0A}, {0x60, 0x0C}, {0x60, 0x0E}, {0x60, 0x10}, {0x60, 0x12}, {0x60, 0x14}, {0x60, 0x16},
    {0x70, 0x00}, {0x70, 0x02}, {0x70, 0x04}, {0x70, 0x06}, {0x70, 0x08}, {0x70, 0x0A}, {0x70, 0x0C}, {0x70, 0x0E}, {0x70, 0x10}, {0x70, 0x12}, {0x70, 0x14}, {0x70, 0x16},
    {0x80, 0x00}, {0x80, 0x02}, {0x80, 0x04}, {0x80, 0x06}, {0x80, 0x08}, {0x80, 0x0A}, {0x80, 0x0C}, {0x80, 0x0E}, {0x80, 0x10}, {0x80, 0x12}, {0x80, 0x14}, {0x80, 0x16},
    {0x90, 0x00}, {0x90, 0x02}, {0x90, 0x04}, {0x90, 0x06}, {0x90, 0x08}, {0x90, 0x0A}, {0x90, 0x0C}, {0x90, 0x0E}, {0x90, 0x10}, {0x90, 0x12}, {0x90, 0x14}, {0x90, 0x16},
    {0xA0, 0x00}, {0xA0, 0x02}, {0xA0, 0x04}, {0xA0, 0x06}, {0xA0, 0x08}, {0xA0, 0x0A}, {0xA0, 0x0C}, {0xA0, 0x0E}, {0xA0, 0x10}, {0xA0, 0x12}, {0xA0, 0x14}, {0xA0, 0x16}
};

bool load_song_data(const char *file_name);
bool load_wave_kit(const char* file_name);

void write_opl(const bool second_opl_port, const uint8_t reg, const uint8_t value);
uint8_t read_opl(const bool second_opl_port, const uint8_t reg);

void update_instrument_4op(const uint8_t channel, const uint8_t instrument);
void update_instrument_2op(const uint8_t channel, const uint8_t instrument);

void update_pan_setting_4op(const uint8_t channel, const uint8_t pan_setting);
void update_pan_setting_2op(const uint8_t channel, const uint8_t pan_setting);

bool next_song_position();
void calculate_wave(const uint8_t channel);
void play_waves();

void tempo_command();
void pattern_end_command();
void status_byte_command();
void transpose_command();

void handle_frequency_mode();
void fm_handle_pitch_bend(const uint8_t channel, MFM_FM_PORT_DATA *port_data, MFM_FM_STATUS_TABLE *status_table);
void fm_handle_modulation(const uint8_t channel, MFM_FM_PORT_DATA *port_data, MFM_FM_STATUS_TABLE *status_table);
void wave_handle_pitch_bend(const uint8_t channel);
void wave_handle_modulation(const uint8_t channel);

void fm_note_on_event(const uint8_t channel, MFM_FM_PORT_DATA *port_data, MFM_FM_STATUS_TABLE *status_table);
void fm_note_off_event(const uint8_t channel, MFM_FM_PORT_DATA *port_data, MFM_FM_STATUS_TABLE *status_table);
void fm_4op_instrument_change_event(const uint8_t channel);
void fm_2op_instrument_change_event(const uint8_t channel);
void fm_4op_volume_change_event(const uint8_t channel, uint8_t volume_value);
void fm_2op_volume_change_event(const uint8_t channel, uint8_t volume_value);
void fm_4op_stereo_change_event(const uint8_t channel, const uint8_t stereo_value);
void fm_2op_stereo_change_event(const uint8_t channel, const uint8_t stereo_value);
void fm_note_link_event(const uint8_t channel, MFM_FM_PORT_DATA *port_data, MFM_FM_STATUS_TABLE* status_table, const uint8_t link_value);
void fm_pitch_bend_event(const uint8_t channel, MFM_FM_STATUS_TABLE* status_table, const uint8_t pitch_bend_value);
void fm_4op_brightness_event(const uint8_t channel, const uint8_t brightness_value);
void fm_2op_brightness_event(const uint8_t channel, const uint8_t brightness_value);
void fm_detune_event(const uint8_t channel, MFM_FM_STATUS_TABLE *status_table, const uint8_t detune_value);
void fm_modulation_event(const uint8_t channel, MFM_FM_STATUS_TABLE *status_table, const uint8_t modulation_value);

void wave_note_on_event(const uint8_t channel);
void wave_note_off_event(const uint8_t channel);
void wave_change_event(const uint8_t channel);
void wave_volume_change_event(const uint8_t channel);
void wave_stereo_change_event(const uint8_t channel);
void wave_note_link_event(const uint8_t channel);
void wave_pitch_bend_event(const uint8_t channel);
void wave_detune_event(const uint8_t channel);
void wave_modulation_event(const uint8_t channel);
void wave_damp_event(const uint8_t channel);

#endif /* __MFM_H */
