/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * mid_fm.h
 *
 * MID: Standard MIDI file player (FM)
 */

#ifndef __MID_FM_H
#define __MID_H

#include <stdint.h>

#define MIDI_HEADER_SIZE 14

#define MAX_NR_OF_TRACKS 64

#define NR_OF_MIDI_CHANNELS 16
#define NR_OF_FM_CHANNELS 15
#define NR_OF_PERCUSSION_CHANNELS 3

#define NR_OF_INSTRUMENTS 128

#define DRUM_DATA_SIZE 9
#define NR_OF_DRUM_DATA 3

#define AM  0x80
#define VIB 0x00

#define PAN_SETTING_LEFT  0x10
#define PAN_SETTING_MID   0x30
#define PAN_SETTING_RIGHT 0x20

typedef struct 
{
    uint8_t   file_format;
    uint16_t  number_of_tracks;
    uint16_t  ticks_per_qnote;
} MIDI_HEADER;

typedef struct 
{
    bool      track_finished;
    uint32_t  waiting_for;
    uint16_t  length;

    uint8_t   last_command;

    uint8_t   start_segment;
    uint8_t  *start_track_data;
    uint8_t   segment;
    uint8_t  *track_data;
} TRACK_DATA;

typedef struct
{
    uint32_t ticks_per_qnote;
    uint32_t clock_ticks;
    uint32_t tick_counter;
    uint32_t midi_counter;
} TIME_DATA;

typedef struct
{
    uint8_t instrument;
    uint8_t panning;
    uint8_t fm_panning;
} MIDI_CHANNEL_DATA;

typedef struct
{
    bool in_use;
    
    uint32_t activated;

    uint8_t midi_link;
    uint8_t note_number;
    uint8_t note_velocity;
} FM_CHANNEL_DATA;

const uint8_t g_op_addr[9] = 
{
    0, 1, 2, 8, 9, 10, 16, 17, 18
};

const uint16_t g_fnumber[128] = 
{
     86,  
     91,  96,  102, 108, 114, 121, 128, 136, 144,
     153, 162, 172, 182, 192, 204, 108, 114, 121,
     128, 136, 144, 153, 162, 172, 182, 192, 204,
     216, 229, 242, 257, 136, 144, 153, 162, 172,
     182, 192, 204, 216, 229, 242, 257, 272, 288,
     306, 324, 172, 182, 192, 204, 216, 229, 242,
     257, 272, 288, 306, 324, 343, 363, 385, 408,
     216, 229, 242, 257, 272, 288, 306, 324, 343,
     363, 385, 408, 432, 458, 485, 514, 272, 288,
     306, 324, 343, 363, 385, 408, 432, 458, 485,
     514, 544, 577, 611, 647, 343, 363, 385, 408,
     432, 458, 485, 514, 544, 577, 611, 647, 686,
     726, 770, 816, 432, 458, 485, 514, 544, 577,
     611, 647, 686, 726, 770, 816, 864, 916, 970, 
     1023
};

const uint8_t real_voice[NR_OF_FM_CHANNELS] = 
{
     0, 1, 2, 3, 4, 5, 9, 10, 11, 12, 13, 14, 15, 16, 17
};

static const uint8_t g_drum_registers[NR_OF_DRUM_DATA][DRUM_DATA_SIZE] =
{
    {0x30, 0x33, 0x50, 0x53, 0x70, 0x73, 0x90, 0x93, 0xC6},
    {0x31, 0x34, 0x51, 0x54, 0x71, 0x74, 0x91, 0x94, 0xC7},
    {0x32, 0x35, 0x52, 0x55, 0x72, 0x75, 0x92, 0x95, 0xC8}
};

const uint8_t g_drum_data[NR_OF_DRUM_DATA][DRUM_DATA_SIZE] =
{
    { 0x01, 0x01, 0x0A, 0x02, 0xDF, 0xF8, 0x6A, 0x6D, PAN_SETTING_MID },     /* BD      */
    { 0x01, 0x01, 0x0A, 0x02, 0xC8, 0xD8, 0xA7, 0x48, PAN_SETTING_MID },     /* SD / HH */
    { 0x05, 0x01, 0x08, 0x08, 0xF8, 0xAA, 0x59, 0x55, PAN_SETTING_MID }      /* TM / TC */
};

MIDI_HEADER g_header;
TRACK_DATA  g_track_data[MAX_NR_OF_TRACKS];

uint8_t g_number_of_tracks;
uint8_t g_track;
uint8_t g_byte_bd;

uint32_t g_qnote_duration;

uint32_t g_ticks_per_update;
uint32_t g_MIDI_counter;

MIDI_CHANNEL_DATA g_midi_channel_data[NR_OF_MIDI_CHANNELS];
FM_CHANNEL_DATA   g_fm_channel_data[NR_OF_FM_CHANNELS];

uint8_t g_note_off_data[NR_OF_FM_CHANNELS][2];

const uint8_t M2[NR_OF_INSTRUMENTS] =
{
    0x02, 0x82, 0x42, 0x04, 0x82, 0x04, 0x02, 0x08, 0x88, 0x0C, 0x02, 0xC2, 0x8C, 0x02, 0x8C, 0x82, 
    0x8C, 0x62, 0x84, 0x04, 0x04, 0x04, 0x04, 0x04, 0x02, 0x02, 0x84, 0x02, 0xC2, 0x08, 0x02, 0x82, 
    0xE2, 0xC2, 0x02, 0x22, 0x42, 0x02, 0xC8, 0x4C, 0x04, 0xA2, 0x04, 0x08, 0x08, 0x08, 0xA4, 0x02,
    0x02, 0x02, 0x02, 0x82, 0x84, 0x08, 0x02, 0x02, 0x04, 0x02, 0x02, 0xC4, 0x02, 0x02, 0x02, 0x04,
    0x02, 0x04, 0x02, 0x02, 0xC2, 0x02, 0x02, 0x44, 0x02, 0x44, 0x02, 0x42, 0x44, 0x04, 0x02, 0x02, 
    0x04, 0x04, 0x02, 0x02, 0x82, 0x04, 0x02, 0x02, 0x08, 0x02, 0x02, 0x08, 0x02, 0x44, 0xE2, 0x82,
    0x02, 0x0C, 0xC8, 0xC4, 0x42, 0xC2, 0xE2, 0x2C, 0xE2, 0xE4, 0xF0, 0x18, 0xC2, 0x04, 0x7C, 0x38, 
    0x14, 0x14, 0x14, 0x14, 0x08, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x82, 0x82, 0x02
};

const uint8_t M4[NR_OF_INSTRUMENTS] =
{
    0x30, 0x05, 0x08, 0x01, 0x20, 0x20, 0x10, 0x07, 0x20, 0x07, 0x3F, 0x20, 0x0D, 0x21, 0x20, 0x03,
    0x0F, 0x1F, 0x1C, 0x07, 0x07, 0x1A, 0x0F, 0x00, 0x10, 0x0D, 0x11, 0x10, 0x20, 0x2F, 0x03, 0x10,
    0x00, 0x00, 0x20, 0x05, 0x17, 0x20, 0x20, 0x03, 0x0D, 0x17, 0x05, 0x05, 0x05, 0x05, 0x17, 0x10,
    0x20, 0x09, 0x20, 0x08, 0x20, 0x20, 0x20, 0x00, 0x1C, 0x20, 0x00, 0x11, 0x0E, 0x0E, 0x00, 0x10,
    0x08, 0x10, 0x04, 0x04, 0x17, 0x04, 0x13, 0x17, 0x3F, 0x20, 0x3F, 0x21, 0x20, 0x01, 0x3F, 0x01,
    0x08, 0x08, 0x17, 0x00, 0x10, 0x20, 0x00, 0x08, 0x20, 0x20, 0x08, 0x20, 0x10, 0x32, 0x10, 0x10,
    0x00, 0x02, 0x1C, 0x00, 0x20, 0x30, 0x30, 0x30, 0x30, 0x3C, 0x05, 0x31, 0x32, 0x10, 0x3B, 0x1B,
    0x05, 0x05, 0x05, 0x0F, 0x11, 0x00, 0x3F, 0x00, 0x0E, 0x00, 0x00, 0x3F, 0x3F, 0x00, 0x00, 0x00
};

const uint8_t M6[NR_OF_INSTRUMENTS] =
{
    0xF1, 0xF1, 0xF1, 0xA1, 0xF3, 0xA1, 0x71, 0xF7, 0xF1, 0xF7, 0x00, 0xA1, 0x87, 0xF1, 0x23, 0x88,
    0x87, 0x16, 0xFF, 0x85, 0x87, 0x47, 0xFB, 0x47, 0xF4, 0xF3, 0xF2, 0xF4, 0xF1, 0xF1, 0x75, 0xA3,
    0xF1, 0xF9, 0xF1, 0x81, 0xF5, 0x51, 0xF4, 0xF5, 0x8C, 0xAE, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0x13,
    0x51, 0x31, 0xF1, 0xF1, 0x11, 0x11, 0x11, 0x55, 0x55, 0x45, 0x24, 0x5A, 0x54, 0x54, 0xF6, 0xF4,
    0xAA, 0xF4, 0xF4, 0xF4, 0xFF, 0x54, 0xF1, 0xF1, 0x00, 0x55, 0xFF, 0xA1, 0x33, 0xF1, 0x00, 0xF1,
    0xF1, 0xE2, 0xF1, 0x81, 0xF1, 0x11, 0x81, 0x81, 0x11, 0x81, 0x44, 0x11, 0xF1, 0xD2, 0xF4, 0x11,
    0xF1, 0xFD, 0x92, 0x22, 0x55, 0xF1, 0xF1, 0xF1, 0xF1, 0x7B, 0x2A, 0x33, 0x74, 0x74, 0x1A, 0x45,
    0xF5, 0xF5, 0x45, 0x7C, 0xF5, 0xF1, 0x61, 0xF1, 0xF1, 0xF1, 0xF1, 0x00, 0xF1, 0x11, 0x11, 0xFF
};

const uint8_t M8[NR_OF_INSTRUMENTS] =
{
    0x84, 0x0C, 0x01, 0x21, 0x03, 0x01, 0xE1, 0x07, 0x03, 0x07, 0x00, 0x25, 0x81, 0x15, 0x8F, 0x1C,
    0x87, 0x14, 0x24, 0x87, 0x87, 0x87, 0x01, 0x87, 0xF1, 0x41, 0x25, 0xF1, 0xF1, 0x74, 0xA1, 0xF3,
    0xF1, 0xF1, 0xF1, 0x11, 0x03, 0x51, 0x04, 0xD3, 0x0B, 0x19, 0x05, 0x05, 0x05, 0x4C, 0x03, 0x11,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x04, 0x05, 0x05, 0x04, 0x24, 0x21, 0x21, 0x07, 0x03,
    0x1A, 0x04, 0x04, 0x04, 0x01, 0x04, 0x11, 0x11, 0x00, 0xF1, 0x05, 0x21, 0xF1, 0x01, 0x00, 0x08,
    0x11, 0x01, 0x01, 0x81, 0x03, 0x01, 0x81, 0x01, 0x01, 0x81, 0x83, 0x01, 0x41, 0xD2, 0xF5, 0xF2,
    0x01, 0xC4, 0x52, 0x83, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0x33, 0x4F, 0x03, 0x15, 0x05, 0x9E, 0x9F,
    0x05, 0x05, 0x43, 0xDD, 0x04, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01, 0x15
};

const uint8_t ME[NR_OF_INSTRUMENTS] =
{
    0x00, 0x00, 0x01, 0x01, 0x00, 0x01, 0x03, 0x03, 0x00, 0x03, 0x00, 0x00, 0x03, 0x02, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x01, 0x03, 0x00, 0x00, 0x01, 0x01, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x01, 0x02, 0x01, 0x01, 0x01, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x01,
    0x00, 0x00, 0x02, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x01, 0x01, 0x00, 0x02, 0x03,
    0x01, 0x01, 0x01, 0x03, 0x00, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

const uint8_t MC[NR_OF_INSTRUMENTS] =
{
    0x00, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x02, 0x06, 0x04, 0x08, 0x04, 0x04,
    0x06, 0x0E, 0x00, 0x06, 0x06, 0x06, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x02, 0x08, 0x08, 0x04,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x08, 0x04, 0x0E, 0x0E, 0x04, 0x04, 0x00, 0x0C, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x06, 0x00, 0x06, 0x0A, 0x04, 0x0C,
    0x0A, 0x04, 0x04, 0x04, 0x04, 0x04, 0x08, 0x00, 0x04, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x02,
    0x0E, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x02, 0x04, 0x04, 0x0E, 0x04, 0x0A, 0x0A,
    0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x04, 0x08, 0x0E, 0x0E, 0x00, 0x00, 0x0E, 0x0E, 0x0E
};

const uint8_t C2[NR_OF_INSTRUMENTS] =
{
    0x02, 0x04, 0x04, 0x42, 0x44, 0x04, 0xC4, 0x08, 0x84, 0x64, 0x04, 0xC2, 0x04, 0x04, 0x44, 0x02,
    0x44, 0xC4, 0x42, 0x42, 0xC4, 0x82, 0x42, 0xC2, 0x82, 0x02, 0xCC, 0xC2, 0x04, 0x82, 0x82, 0x82,
    0x62, 0x42, 0xC2, 0x22, 0x02, 0xC2, 0x02, 0x02, 0x42, 0x10, 0x44, 0x14, 0x44, 0x44, 0xA4, 0x02,
    0xC2, 0x42, 0x42, 0x44, 0xC4, 0x44, 0x44, 0x82, 0xC2, 0x42, 0x44, 0x42, 0x48, 0x48, 0x44, 0x44,
    0x04, 0x02, 0x02, 0x02, 0xC2, 0x02, 0x04, 0x48, 0x42, 0xC4, 0x42, 0x44, 0xC4, 0x02, 0xC4, 0x04,
    0x02, 0x02, 0x02, 0x02, 0x02, 0x42, 0x02, 0x02, 0x44, 0x02, 0x02, 0x44, 0x02, 0xA8, 0x04, 0xC4,
    0x02, 0x68, 0x44, 0x04, 0x42, 0xC2, 0xE8, 0xA8, 0xEC, 0xE8, 0x2C, 0x3C, 0xC4, 0x42, 0xF0, 0x48,
    0x14, 0x14, 0x14, 0xB0, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x08, 0x88, 0x82, 0x82, 0x02
};

const uint8_t C4[NR_OF_INSTRUMENTS] =
{
    0x00, 0x00, 0x05, 0x06, 0x00, 0x03, 0x08, 0x07, 0x07, 0x07, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03,
    0x00, 0x00, 0x00, 0x03, 0x03, 0x03, 0x03, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x05, 0x03, 0x00, 0x05, 0x05, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x09, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x03, 0x03, 0x08, 0x00, 0x0A, 0x06, 0x0C,
    0x00, 0x04, 0x04, 0x00, 0x05, 0x04, 0x0B, 0x03, 0x03, 0x03, 0x01, 0x03, 0x03, 0x08, 0x05, 0x01,
    0x08, 0x0A, 0x03, 0x03, 0x00, 0x00, 0x00, 0x08, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x03, 0x04, 0x05,
    0x05, 0x05, 0x03, 0x05, 0x0B, 0x03, 0x00, 0x00, 0x08, 0x10, 0x10, 0x00, 0x00, 0x08, 0x08, 0x00
};

const uint8_t C6[NR_OF_INSTRUMENTS] =
{
    0xF5, 0xA5, 0xF4, 0xF4, 0xF3, 0xA2, 0x62, 0xF7, 0xF1, 0xFE, 0xF5, 0xA1, 0xF5, 0xA7, 0x43, 0x41,
    0xA4, 0xA5, 0x81, 0xA5, 0x87, 0x42, 0x41, 0x43, 0xF5, 0xF4, 0x65, 0xF4, 0xF5, 0xF4, 0xA1, 0xF3,
    0xF2, 0x73, 0xF3, 0xF1, 0xF3, 0xFF, 0xF1, 0xF5, 0x53, 0x8F, 0xF5, 0xF5, 0xF5, 0xF2, 0xF5, 0xA3,
    0xA5, 0x51, 0xF2, 0xF1, 0xA1, 0xA1, 0xA1, 0xD4, 0xFF, 0xF1, 0x28, 0x71, 0x54, 0x51, 0xF4, 0xFE,
    0xF5, 0xF4, 0xF4, 0xF8, 0xFF, 0xA8, 0xF1, 0x31, 0xF1, 0xF1, 0xFF, 0x33, 0xF1, 0xF1, 0xA1, 0x22,
    0xF1, 0xF1, 0xF3, 0xF3, 0xF1, 0xA1, 0x81, 0xF1, 0xA1, 0x81, 0x81, 0xA1, 0x81, 0xE2, 0x91, 0x81,
    0xFF, 0xF1, 0x82, 0x51, 0xF1, 0xF1, 0xF1, 0xD1, 0x72, 0x55, 0xF5, 0x6C, 0x43, 0x31, 0x7C, 0xF5,
    0xF1, 0xF5, 0xF5, 0xF5, 0xF1, 0xF1, 0xF1, 0x2F, 0xF1, 0xF1, 0xFF, 0x3F, 0x55, 0x11, 0x11, 0xF5
};

const uint8_t C8[NR_OF_INSTRUMENTS] =
{
    0x04, 0x35, 0xC4, 0x43, 0x03, 0x02, 0xE2, 0x07, 0x03, 0x23, 0x16, 0x25, 0x2D, 0x16, 0x1E, 0x11,
    0x18, 0x25, 0x83, 0x22, 0x34, 0x83, 0x17, 0x83, 0x04, 0x03, 0xD5, 0xF1, 0x74, 0x74, 0x53, 0xF3,
    0xA1, 0x93, 0xF3, 0x11, 0x03, 0x12, 0xB6, 0x05, 0x53, 0x05, 0x15, 0x05, 0x17, 0x05, 0x03, 0x03,
    0x02, 0x11, 0x02, 0x02, 0x22, 0x52, 0xA2, 0x04, 0x05, 0x05, 0x24, 0x24, 0x23, 0x14, 0x04, 0x02,
    0x03, 0x04, 0x04, 0x04, 0x05, 0x11, 0x61, 0x22, 0x05, 0x05, 0x07, 0x13, 0x25, 0x01, 0x54, 0x01,
    0x01, 0x04, 0x81, 0x01, 0x01, 0xA2, 0x81, 0x01, 0xA2, 0x81, 0x83, 0xA2, 0xC1, 0xB2, 0xF1, 0xF2,
    0x01, 0x11, 0x92, 0x62, 0xF1, 0xF1, 0xF1, 0xF6, 0x74, 0x15, 0x05, 0x78, 0x95, 0x13, 0x7C, 0x05,
    0x04, 0x05, 0x05, 0x05, 0x01, 0x05, 0x06, 0x07, 0x03, 0x02, 0x01, 0x1A, 0x31, 0x01, 0x01, 0x07
};

const uint8_t CE[NR_OF_INSTRUMENTS] =
{
    0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
    0x01, 0x03, 0x03, 0x00, 0x03, 0x00, 0x03, 0x00, 0x02, 0x02, 0x01, 0x00, 0x01, 0x03, 0x03, 0x00,
    0x00, 0x02, 0x02, 0x00, 0x00, 0x02, 0x01, 0x01, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x01, 0x02, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x02, 0x00, 0x01, 0x01, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x03, 0x00, 0x01, 0x00, 0x00, 0x02, 0x03, 0x02, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x03, 0x00, 0x02, 0x00, 0x00, 0x00
};

void load_track_data();

void enable_drums();

void change_speed(uint32_t speed);

uint8_t read_byte();
uint32_t get_variable_len();
void get_delta_time();

void handle_track_event();
void handle_meta_event(uint8_t sub_type);
void handle_sys_ex_event();

void write_opl_control(uint8_t reg, uint8_t fm_channel, uint8_t value);
void write_opl_operator(uint8_t reg, uint8_t fm_channel, uint8_t value);

void enable_note(uint8_t voice, uint8_t note);
void disable_note(uint8_t voice);
void cut_note(uint8_t voice);

void set_instrument(uint8_t voice, uint8_t instrument, uint8_t volume, uint8_t panning);
void set_drum(uint8_t fm_channel, uint8_t o2, uint8_t o4, uint8_t o6, uint8_t o8, uint8_t oE, uint8_t oC);

void note_off(uint8_t midi_channel, uint8_t note, uint8_t velocity);
void note_on(uint8_t midi_channel, uint8_t note, uint8_t velocity);
void drum_on(uint8_t number, uint8_t velocity);
void drum_off(uint8_t number, uint8_t velocity);
void pitch_wheel(uint8_t midi_channel, uint8_t vlue);
void control_change(uint8_t midi_channel, uint8_t id, uint8_t value);
void channel_pressure(uint8_t midi_channel, uint8_t pressure);
void key_pressure(uint8_t midi_channel, uint8_t note, uint8_t pressure);
void program_change(uint8_t midi_channel, uint8_t program);

#endif /* __MID_FM_H */