/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * mod.h
 *
 * MOD: Amiga MOD player
 */

#ifndef __MOD_H
#define __MOD_H

#define OPL_WAVE_MEMORY  0x200000

#define MAX_PATTERN  128

#define MAX_PATTERN_STEP 63
#define STEP_DATA_SIZE   4

#define MAX_NR_OF_CHANNELS 24

#define NR_OF_SAMPLES 31

#define MAX_VOLUME 64

#define CHANNEL_STEP_SIZE 4

#define NOTE_NUMBERS 36

typedef struct
{
    char*   id;
    uint8_t nr_of_channels;
    char*   description;
} MOD_FILE_FORMAT;

typedef struct
{
    char     sample_name[22];
    uint16_t sample_length;
    uint8_t  finetune;
    uint8_t  volume;
    uint16_t sample_repeat_point;
    uint16_t sample_repeat_length;
} MOD_SAMPLE_INFO;

typedef struct
{
    char song_name[20];
    
    MOD_SAMPLE_INFO sample_info[31];

    uint8_t song_length;
    uint8_t dummy;
    uint8_t pattern_table[128];

    char    module_type[4];
} MOD_HEADER;

typedef struct
{
    uint8_t  segment;
    uint8_t *address;
} MOD_PATTERN_INFO;

typedef struct
{
    uint8_t  sample_number;
    uint8_t  old_sample_number;
    uint8_t  note_number;
    uint16_t period;
    uint8_t  volume;
    uint16_t temp_command;
    uint8_t  pitch_bend_speed;
    uint8_t  vibrato_command;
    uint8_t  vibrato_table_position;
    uint8_t  tremolo_command;
    uint8_t  tremolo_table_position;
    uint8_t  wave_control;
    uint8_t  pattern_loop_start;
    uint8_t  pattern_loop_count;
} MOD_CHANNEL_INFO;

typedef struct 
{
    uint8_t  sample_number;
    uint16_t period;
    uint8_t  effect_command;
    uint8_t  effect_data;
    uint8_t  extended_command;
} MOD_STEP_DATA;

static const MOD_FILE_FORMAT g_file_formats[] =
{
    {"M.K.",  4, "Standard 4CH module"},
    {"M!K!",  4, "4CH module extended length"},
    {"FLT4",  4, "StarTrekker 4CH module"},
    {"FLT8",  8, "StarTrekker 8CH module"},
    {"2CHN",  2, "FastTracker 2CH module"},
    {"4CHN",  4, "FastTracker 4CH module"},
    {"6CHN",  6, "FastTracker 6CH module"},
    {"8CHN",  8, "FastTracker 8CH module"},
    {"16CH", 16, "FastTracker 16CH module"},
    {"OCTA",  8, "Octalyzer module"}
};

static const uint16_t g_periods[NOTE_NUMBERS] =
{
    /* Tuning 0, Normal */
    856,808,762,720,678,640,604,570,538,508,480,453,
    428,404,381,360,339,320,302,285,269,254,240,226,
    214,202,190,180,170,160,151,143,135,127,120,113
};

static const int8_t g_finetune_table[][NOTE_NUMBERS] =
{
    /* Tuning 1 */
    {-6, -6, -5, -5, -4, -3, -3, -3, -3, -3, -3, -3,
     -3, -3, -2, -3, -2, -2, -2, -1, -1, -1, -1, -1,
     -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, +0},
    /* Tuning 2 */
    {-12, -12,-10,-11,-8, -8, -7, -7, -6, -6, -6, -6,
     -6, -6, -5, -5, -4, -4, -4, -3, -3, -3, -3, -2,
     -3, -3, -2, -3, -3, -2, -2, -2, -2, -2, -2, -1},
    /* Tuning 3 */
    {-18,-17,-16,-16,-13,-12,-12,-11,-10,-10,-10,-9,
     -9, -9, -8, -8, -7, -6, -6, -5, -5, -5, -5, -4,
     -5, -4, -3, -4, -4, -3, -3, -3, -3, -2, -2, -2},
    /* Tuning 4 */
    {-24,-23,-21,-21,-18,-17,-16,-15,-14,-13,-13,-12,
     -12,-12,-11,-10,-9, -8, -8, -7, -7, -7, -7, -6,
     -6, -6, -5, -5, -5, -4, -4, -4, -4, -3, -3, -3},
    /* Tuning 5 */
    {-30,-29,-26,-26,-23,-21,-20,-19,-18,-17,-17,-16,
     -15,-14,-13,-13,-11,-11,-10,-9, -9, -9, -8, -7,
     -8, -7, -6, -6, -6, -5, -5, -5, -5, -4, -4, -4},
    /* Tuning 6 */
    {-36,-34,-32,-31,-27,-26,-24,-23,-22,-21,-20,-19,
     -18,-17,-16,-15,-14,-13,-12,-11,-11,-10,-10,-9,
     -9, -9, -7, -8, -7, -6, -6, -6, -6, -5, -5, -4},
    /* Tuning 7 */
    {-42,-40,-37,-36,-32,-30,-29,-27,-25,-24,-23,-22,
     -21,-20,-18,-18,-16,-15,-14,-13,-13,-12,-12,-10,
     -10,-10,-9, -9, -9, -8, -7, -7, -7, -6, -6, -5},
    /* Tuning -8 */
    {+51,+48,+46,+42,+42,+38,+36,+34,+32,+30,+28,+27,
     +25,+24,+23,+21,+21,+19,+18,+17,+16,+15,+14,+14,
     +12,+12,+12,+10,+10,+10,+9, +8, +8, +8, +7, +7},
    /* Tuning -7 */
    {+44,+42,+40,+37,+37,+35,+32,+31,+29,+27,+25,+24,
     +22,+21,+20,+19,+18,+17,+16,+15,+15,+14,+13,+12,
     +11,+10,+10,+9, +9, +9, +8, +7, +7, +7, +6, +6},
    /* Tuning -6 */
    {+38,+36,+34,+32,+31,+30,+28,+27,+25,+24,+22,+21,
     +19,+18,+17,+16,+16,+15,+14,+13,+13,+12,+11,+11,
     +9, +9, +9, +8, +7, +7, +7, +6, +6, +6, +5, +5},
    /* Tuning -5 */
    {+31,+30,+29,+26,+26,+25,+24,+22,+21,+20,+18,+17,
     +16,+15,+14,+13,+13,+12,+12,+11,+11,+10,+9, +9,
     +8, +7, +8, +7, +6, +6, +6, +5, +5, +5, +5, +5},
    /* Tuning -4 */
    {+25,+24,+23,+21,+21,+20,+19,+18,+17,+16,+14,+14,
     +13,+12,+11,+10,+11,+10,+10,+9, +9, +8, +7, +7,
     +6, +6, +6, +5, +5, +5, +5, +4, +4, +4, +3, +4},
    /* Tuning -3 */
    {+19,+18,+17,+16,+16,+15,+15,+14,+13,+12,+11,+10,
     +9, +9, +9, +9, +8, +8, +7, +7, +7, +6, +5, +6,
     +5, +4, +5, +4, +4, +4, +4, +3, +3, +3, +3, +3},
    /* Tuning -2 */
    {+12,+12,+12,+10,+11,+11,+10,+10,+9, +8, +7, +7,
     +6, +6, +6, +5, +6, +5, +5, +5, +5, +4, +4, +4,
     +3, +3, +3, +3, +2, +3, +3, +2, +2, +2, +2, +2},
    /* Tuning -1 */
    {+6, +6, +6, +5, +6, +6, +6, +5, +5, +5, +4, +4,
     +3, +3, +3, +3, +3, +3, +3, +3, +3, +2, +2, +2,
     +2, +1, +2, +1, +1, +1, +1, +1, +1, +1, +1, +1}
};

static const uint8_t g_pitch_table[] =
{
    0x06,0x04,0xEA,0x03,0xCE,0x03,0xB2,0x03,0x98,0x03,0x7E,0x03,0x64,0x03,0x4A,0x03,0x32,0x03,0x18,0x03,0x00,0x03,0xE8,0x02,
    0xD2,0x02,0xBA,0x02,0xA4,0x02,0x8E,0x02,0x78,0x02,0x62,0x02,0x4E,0x02,0x38,0x02,0x24,0x02,0x10,0x02,0xFC,0x01,0xE8,0x01,
    0xD6,0x01,0xC2,0x01,0xB0,0x01,0x9E,0x01,0x8C,0x01,0x7A,0x01,0x68,0x01,0x56,0x01,0x46,0x01,0x34,0x01,0x24,0x01,0x14,0x01,
    0x04,0x01,0xF4,0x00,0xE4,0x00,0xD4,0x00,0xC6,0x00,0xB6,0x00,0xA8,0x00,0x98,0x00,0x8A,0x00,0x7C,0x00,0x6E,0x00,0x60,0x00,
    0x52,0x00,0x44,0x00,0x38,0x00,0x2A,0x00,0x1C,0x00,0x10,0x00,0x04,0x00,0xEE,0xF7,0xD6,0xF7,0xBC,0xF7,0xA4,0xF7,0x8C,0xF7,
    0x74,0xF7,0x5E,0xF7,0x46,0xF7,0x30,0xF7,0x18,0xF7,0x02,0xF7,0xEC,0xF6,0xD6,0xF6,0xC2,0xF6,0xAC,0xF6,0x96,0xF6,0x82,0xF6,
    0x6E,0xF6,0x58,0xF6,0x44,0xF6,0x30,0xF6,0x1C,0xF6,0x0A,0xF6,0xF6,0xF5,0xE2,0xF5,0xD0,0xF5,0xBE,0xF5,0xAA,0xF5,0x98,0xF5,
    0x86,0xF5,0x74,0xF5,0x62,0xF5,0x50,0xF5,0x40,0xF5,0x2E,0xF5,0x1E,0xF5,0x0C,0xF5,0xFC,0xF4,0xEC,0xF4,0xDA,0xF4,0xCA,0xF4,
    0xBA,0xF4,0xAA,0xF4,0x9A,0xF4,0x8C,0xF4,0x7C,0xF4,0x6C,0xF4,0x5E,0xF4,0x4E,0xF4,0x40,0xF4,0x30,0xF4,0x22,0xF4,0x14,0xF4,
    0x06,0xF4,0xF8,0xF3,0xEA,0xF3,0xDC,0xF3,0xCE,0xF3,0xC0,0xF3,0xB2,0xF3,0xA4,0xF3,0x98,0xF3,0x8A,0xF3,0x7E,0xF3,0x70,0xF3,
    0x64,0xF3,0x56,0xF3,0x4A,0xF3,0x3E,0xF3,0x32,0xF3,0x24,0xF3,0x18,0xF3,0x0C,0xF3,0x00,0xF3,0xF4,0xF2,0xE8,0xF2,0xDE,0xF2,
    0xD2,0xF2,0xC6,0xF2,0xBA,0xF2,0xB0,0xF2,0xA4,0xF2,0x9A,0xF2,0x8E,0xF2,0x84,0xF2,0x78,0xF2,0x6E,0xF2,0x62,0xF2,0x58,0xF2,
    0x4E,0xF2,0x44,0xF2,0x38,0xF2,0x2E,0xF2,0x24,0xF2,0x1A,0xF2,0x10,0xF2,0x06,0xF2,0xFC,0xF1,0xF2,0xF1,0xE8,0xF1,0xE0,0xF1,
    0xD6,0xF1,0xCC,0xF1,0xC2,0xF1,0xBA,0xF1,0xB0,0xF1,0xA6,0xF1,0x9E,0xF1,0x94,0xF1,0x8C,0xF1,0x82,0xF1,0x7A,0xF1,0x70,0xF1,
    0x68,0xF1,0x60,0xF1,0x56,0xF1,0x4E,0xF1,0x46,0xF1,0x3E,0xF1,0x34,0xF1,0x2C,0xF1,0x24,0xF1,0x1C,0xF1,0x14,0xF1,0x0C,0xF1,
    0x04,0xF1,0xFC,0xF0,0xF4,0xF0,0xEC,0xF0,0xE4,0xF0,0xDC,0xF0,0xD4,0xF0,0xCE,0xF0,0xC6,0xF0,0xBE,0xF0,0xB6,0xF0,0xAE,0xF0,
    0xA8,0xF0,0xA0,0xF0,0x98,0xF0,0x92,0xF0,0x8A,0xF0,0x84,0xF0,0x7C,0xF0,0x74,0xF0,0x6E,0xF0,0x66,0xF0,0x60,0xF0,0x5A,0xF0,
    0x52,0xF0,0x4C,0xF0,0x44,0xF0,0x3E,0xF0,0x38,0xF0,0x30,0xF0,0x2A,0xF0,0x24,0xF0,0x1C,0xF0,0x16,0xF0,0x10,0xF0,0x0A,0xF0,
    0x04,0xF0,0xFA,0xE7,0xEE,0xE7,0xE2,0xE7,0xD6,0xE7,0xCA,0xE7,0xBC,0xE7,0xB0,0xE7,0xA4,0xE7,0x98,0xE7,0x8C,0xE7,0x80,0xE7,
    0x74,0xE7,0x6A,0xE7,0x5E,0xE7,0x52,0xE7,0x46,0xE7,0x3A,0xE7,0x30,0xE7,0x24,0xE7,0x18,0xE7,0x0E,0xE7,0x02,0xE7,0xF8,0xE6,
    0xEC,0xE6,0xE2,0xE6,0xD6,0xE6,0xCC,0xE6,0xC2,0xE6,0xB6,0xE6,0xAC,0xE6,0xA2,0xE6,0x96,0xE6,0x8c,0xe6,0x82,0xe6,0x78,0xe6,
    0x6e,0xe6,0x62,0xe6,0x58,0xe6,0x4e,0xe6,0x44,0xE6,0x3A,0xE6,0x30,0xE6,0x26,0xE6,0x1C,0xE6,0x12,0xE6,0x0A,0xE6,0x00,0xE6,
    0xF6,0xE5,0xEC,0xE5,0xE2,0xE5,0xDA,0xE5,0xD0,0xE5,0xC6,0xE5,0xBE,0xE5,0xB4,0xE5,0xAA,0xE5,0xA2,0xE5,0x98,0xE5,0x90,0xE5,
    0x86,0xE5,0x7E,0xE5,0x74,0xE5,0x6C,0xE5,0x62,0xE5,0x5A,0xE5,0x50,0xE5,0x48,0xE5,0x40,0xE5,0x36,0xE5,0x2E,0xE5,0x26,0xE5,
    0x1E,0xE5,0x14,0xE5,0x0C,0xE5,0x04,0xE5,0xFC,0xE4,0xF4,0xE4,0xEC,0xE4,0xE2,0xE4,0xDA,0xE4,0xD2,0xE4,0xCA,0xE4,0xC2,0xE4,
    0xBA,0xE4,0xB2,0xE4,0xAA,0xE4,0xA2,0xE4,0x9A,0xE4,0x94,0xE4,0x8C,0xE4,0x84,0xE4,0x7C,0xE4,0x74,0xE4,0x6C,0xE4,0x64,0xE4,
    0x5E,0xE4,0x56,0xE4,0x4E,0xE4,0x46,0xE4,0x40,0xE4,0x38,0xE4,0x30,0xE4,0x2A,0xE4,0x22,0xE4,0x1A,0xE4,0x14,0xE4,0x0C,0xE4,
    0x06,0xE4,0xFE,0xE3,0xF8,0xE3,0xF0,0xE3,0xEA,0xE3,0xE2,0xE3,0xDC,0xE3,0xD4,0xE3,0xCE,0xE3,0xC6,0xE3,0xC0,0xE3,0xB8,0xE3,
    0xB2,0xE3,0xAC,0xE3,0xA4,0xE3,0x9E,0xE3,0x98,0xE3,0x90,0xE3,0x8A,0xE3,0x84,0xE3,0x7E,0xE3,0x76,0xE3,0x70,0xE3,0x6A,0xE3,
    0x64,0xE3,0x5E,0xE3,0x56,0xE3,0x50,0xE3,0x4A,0xE3,0x44,0xE3,0x3E,0xE3,0x38,0xE3,0x32,0xE3,0x2A,0xE3,0x24,0xE3,0x1E,0xE3,
    0x18,0xE3,0x12,0xE3,0x0C,0xE3,0x06,0xE3,0x00,0xE3,0xFA,0xE2,0xF4,0xE2,0xEE,0xE2,0xE8,0xE2,0xE2,0xE2,0xDE,0xE2,0xD8,0xE2,
    0xD2,0xE2,0xCC,0xE2,0xC6,0xE2,0xC0,0xE2,0xBA,0xE2,0xB6,0xE2,0xB0,0xE2,0xAA,0xE2,0xA4,0xE2,0x9E,0xE2,0x9A,0xE2,0x94,0xE2,
    0x8E,0xE2,0x88,0xE2,0x84,0xE2,0x7E,0xE2,0x78,0xE2,0x72,0xE2,0x6E,0xE2,0x68,0xE2,0x62,0xE2,0x5E,0xE2,0x58,0xE2,0x52,0xE2,
    0x4E,0xE2,0x48,0xE2,0x44,0xE2,0x3E,0xE2,0x38,0xE2,0x34,0xE2,0x2E,0xE2,0x2A,0xE2,0x24,0xE2,0x20,0xE2,0x1A,0xE2,0x16,0xE2,
    0x10,0xE2,0x0C,0xE2,0x06,0xE2,0x02,0xE2,0xFC,0xE1,0xF8,0xE1,0xF2,0xE1,0xEE,0xE1,0xE8,0xE1,0xE4,0xE1,0xE0,0xE1,0xDA,0xE1,
    0xD6,0xE1,0xD0,0xE1,0xCC,0xE1,0xC8,0xE1,0xC2,0xE1,0xBE,0xE1,0xBA,0xE1,0xB4,0xE1,0xB0,0xE1,0xAC,0xE1,0xA6,0xE1,0xA2,0xE1,
    0x9E,0xE1,0x9A,0xE1,0x94,0xE1,0x90,0xE1,0x8C,0xE1,0x88,0xE1,0x82,0xE1,0x7E,0xE1,0x7A,0xE1,0x76,0xE1,0x70,0xE1,0x6C,0xE1,
    0x68,0xE1,0x64,0xE1,0x60,0xE1,0x5C,0xE1,0x56,0xE1,0x52,0xE1,0x4E,0xE1,0x4A,0xE1,0x46,0xE1,0x42,0xE1,0x3E,0xE1,0x3A,0xE1,
    0x34,0xE1,0x30,0xE1,0x2C,0xE1,0x28,0xE1,0x24,0xE1,0x20,0xE1,0x1C,0xE1,0x18,0xE1,0x14,0xE1,0x10,0xE1,0x0C,0xE1,0x08,0xE1,
    0x04,0xE1,0x00,0xE1,0xFC,0xE0,0xF8,0xE0,0xF4,0xE0,0xF0,0xE0,0xEC,0xE0,0xE8,0xE0,0xE4,0xE0,0xE0,0xE0,0xDC,0xE0,0xD8,0xE0,
    0xD4,0xE0,0xD0,0xE0,0xCE,0xE0,0xCA,0xE0,0xC6,0xE0,0xC2,0xE0,0xBE,0xE0,0xBA,0xE0,0xB6,0xE0,0xB2,0xE0,0xAE,0xE0,0xAC,0xE0,
    0xA8,0xE0,0xA4,0xE0,0xA0,0xE0,0x9C,0xE0,0x98,0xE0,0x96,0xE0,0x92,0xE0,0x8E,0xE0,0x8A,0xE0,0x86,0xE0,0x84,0xE0,0x80,0xE0,
    0x7C,0xE0,0x78,0xE0,0x74,0xE0,0x72,0xE0,0x6E,0xE0,0x6A,0xE0,0x66,0xE0,0x64,0xE0,0x60,0xE0,0x5C,0xE0,0x5A,0xE0,0x56,0xE0,
    0x52,0xE0,0x4E,0xE0,0x4C,0xE0,0x48,0xE0,0x44,0xE0,0x42,0xE0,0x3E,0xE0,0x3A,0xE0,0x38,0xE0,0x34,0xE0,0x30,0xE0,0x2E,0xE0,
    0x2A,0xE0,0x26,0xE0,0x24,0xE0,0x20,0xE0,0x1C,0xE0,0x1A,0xE0,0x16,0xE0,0x14,0xE0,0x10,0xE0,0x0C,0xE0,0x0A,0xE0,0x06,0xE0,
    0x04,0xE0,0x00,0xE0,0xFA,0xD7,0xF4,0xD7,0xEE,0xD7,0xE8,0xD7,0xE2,0xD7,0xDC,0xD7,0xD6,0xD7,0xD0,0xD7,0xCA,0xD7,0xC4,0xD7,
    0xBC,0xD7,0xB6,0xD7,0xB0,0xD7,0xAA,0xD7,0xA4,0xD7,0x9E,0xD7,0x98,0xD7,0x92,0xD7,0x8C,0xD7,0x86,0xD7,0x80,0xD7,0x7A,0xD7,
    0x74,0xD7,0x70,0xD7,0x6A,0xD7,0x64,0xD7,0x5E,0xD7,0x58,0xD7,0x52,0xD7,0x4C,0xD7,0x46,0xD7,0x40,0xD7,0x3A,0xD7,0x36,0xD7,
    0x30,0xD7,0x2A,0xD7,0x24,0xD7,0x1E,0xD7,0x18,0xD7,0x14,0xD7,0x0E,0xD7,0x08,0xD7,0x02,0xD7,0xFE,0xD6,0xF8,0xD6,0xF2,0xD6,
    0xEC,0xD6,0xE6,0xD6,0xE2,0xD6,0xDC,0xD6,0xD6,0xD6,0xD2,0xD6,0xCC,0xD6,0xC6,0xD6,0xC2,0xD6,0xBC,0xD6,0xB6,0xD6,0xB2,0xD6,
    0xAC,0xD6,0xA6,0xD6,0xA2,0xD6,0x9C,0xD6,0x96,0xD6,0x92,0xD6,0x8C,0xD6,0x86,0xD6,0x82,0xD6,0x7C,0xD6,0x78,0xD6,0x72,0xD6,
    0x6E,0xD6,0x68,0xD6,0x62,0xD6,0x5E,0xD6,0x58,0xD6,0x54,0xD6,0x4E,0xD6,0x4A,0xD6,0x44,0xD6,0x40,0xD6,0x3A,0xD6,0x36,0xD6,
    0x30,0xD6,0x2C,0xD6,0x26,0xD6,0x22,0xD6,0x1C,0xD6,0x18,0xD6,0x12,0xD6,0x0E,0xD6,0x0A,0xD6,0x04,0xD6,0x00,0xD6,0xFA,0xD5,
    0xF6,0xD5,0xF2,0xD5,0xEC,0xD5,0xE8,0xD5,0xE2,0xD5,0xDE,0xD5,0xDA,0xD5,0xD4,0xD5,0xD0,0xD5,0xCC,0xD5,0xC6,0xD5,0xC2,0xD5,
    0xBE,0xD5,0xB8,0xD5,0xB4,0xD5,0xB0,0xD5,0xAA,0xD5,0xA6,0xD5,0xA2,0xD5,0x9C,0xD5,0x98,0xD5,0x94,0xD5,0x90,0xD5,0x8A,0xD5,
    0x86,0xD5,0x82,0xD5,0x7E,0xD5,0x78,0xD5,0x74,0xD5,0x70,0xD5,0x6C,0xD5,0x66,0xD5,0x62,0xD5,0x5E,0xD5,0x5A,0xD5,0x56,0xD5,
    0x50,0xD5,0x4C,0xD5,0x48,0xD5,0x44,0xD5,0x40,0xD5,0x3C,0xD5,0x36,0xD5,0x32,0xD5,0x2E,0xD5,0x2A,0xD5,0x26,0xD5,0x22,0xD5,
    0x1E,0xD5,0x1A,0xD5,0x14,0xD5,0x10,0xD5,0x0C,0xD5,0x08,0xD5,0x04,0xD5,0x00,0xD5,0xFC,0xD4,0xF8,0xD4,0xF4,0xD4,0xF0,0xD4,
    0xEC,0xD4,0xE8,0xD4,0xE2,0xD4,0xDE,0xD4,0xDA,0xD4,0xD6,0xD4,0xD2,0xD4,0xCE,0xD4,0xCA,0xD4,0xC6,0xD4,0xC2,0xD4,0xBE,0xD4,
    0xBA,0xD4,0xB6,0xD4,0xB2,0xD4,0xAE,0xD4,0xAA,0xD4,0xA6,0xD4,0xA2,0xD4,0x9E,0xD4,0x9A,0xD4,0x96,0xD4,0x94,0xD4,0x90,0xD4,
    0x8C,0xD4,0x88,0xD4,0x84,0xD4,0x80,0xD4,0x7C,0xD4,0x78,0xD4,0x74,0xD4,0x70,0xD4,0x6C,0xD4,0x68,0xD4,0x64,0xD4,0x62,0xD4,
    0x5E,0xD4,0x5A,0xD4,0x56,0xD4,0x52,0xD4,0x4E,0xD4,0x4A,0xD4,0x46,0xD4,0x44,0xD4,0x40,0xD4,0x3C,0xD4,0x38,0xD4,0x34,0xD4,
    0x30,0xD4,0x2E,0xD4,0x2A,0xD4,0x26,0xD4,0x22,0xD4,0x1E,0xD4,0x1A,0xD4,0x18,0xD4,0x14,0xD4,0x10,0xD4,0x0C,0xD4,0x08,0xD4,
    0x06,0xD4,0x02,0xD4,0xFE,0xD3,0xFA,0xD3,0xF8,0xD3,0xF4,0xD3,0xF0,0xD3,0xEC,0xD3,0xEA,0xD3,0xE6,0xD3,0xE2,0xD3,0xDE,0xD3,
    0xDC,0xD3,0xD8,0xD3,0xD4,0xD3,0xD0,0xD3,0xCE,0xD3,0xCA,0xD3,0xC6,0xD3,0xC4,0xD3,0xC0,0xD3,0xBC,0xD3,0xB8,0xD3,0xB6,0xD3,
    0xB2,0xD3,0xAE,0xD3,0xAC,0xD3,0xA8,0xD3,0xA4,0xD3,0xA2,0xD3,0x9E,0xD3,0x9A,0xD3,0x98,0xD3,0x94,0xD3,0x90,0xD3,0x8E,0xD3,
    0x8A,0xD3,0x86,0xD3,0x84,0xD3,0x80,0xD3,0x7E,0xD3,0x7A,0xD3,0x76,0xD3,0x74,0xD3
};

static const uint8_t g_volume_table[] =
{
    0x7F,0x5C,0x52,0x4A,0x44,0x3F,0x3B,0x37,0x34,0x31,
    0x2F,0x2D,0x2A,0x28,0x27,0x25,0x23,0x22,0x20,0x1F,
    0x1E,0x1C,0x1B,0x1A,0x19,0x18,0x17,0x16,0x15,0x14,
    0x13,0x12,0x12,0x11,0x10,0x0F,0x0F,0x0E,0x0D,0x0C,
    0x0C,0x0B,0x0B,0x0A,0x09,0x09,0x08,0x08,0x07,0x06,
    0x06,0x05,0x05,0x04,0x04,0x03,0x03,0x03,0x02,0x02,
    0x01,0x01,0x00,0x00,0x00
};

static const uint8_t g_vibrato_table[] =
{
      0, 24, 49, 74, 97,120,141,161,
    180,197,212,224,235,244,250,253,
    255,253,250,244,235,224,212,197,
    180,161,141,120, 97, 74, 49, 24
};

static const uint8_t g_pan_setting[MAX_NR_OF_CHANNELS] =
{
    10, 6, 6, 10, 10, 6, 6, 10, 10, 6, 6, 10, 10, 6, 6, 10
};

MOD_HEADER       g_mod_header;
MOD_PATTERN_INFO g_pattern_info[MAX_PATTERN];

MOD_CHANNEL_INFO g_channel_info[MAX_NR_OF_CHANNELS];
MOD_STEP_DATA    g_step_data[MAX_NR_OF_CHANNELS];

uint8_t g_module_type;
uint16_t g_pattern_size;

uint8_t g_bpm;
uint8_t g_speed;
uint8_t g_speed_step;

uint8_t g_pattern_step;
uint8_t g_song_position;

uint8_t g_arpeggio;
uint8_t g_command_E5;
uint8_t g_command_EE;

void read_patterns(uint8_t nrofPatterns);
void read_samples();

void next_pattern_step();
void set_pattern_segment();
void update_step_data();

void handle_command(uint8_t channel);
void handle_off_note_command(uint8_t channel);

uint16_t tune_period(uint8_t channel, uint16_t period);
void tone_portamento(uint8_t channel);
void vibrato(uint8_t channel);
void volume_slide(uint8_t channel);
void tremolo(uint8_t channel);

void set_sample(uint8_t channel);
void new_note(uint8_t channel);

void set_sample_number(uint8_t channel, uint8_t sampleNumber);
void set_frequency(uint8_t channel, uint16_t period);
void set_total_level(uint8_t channel, uint8_t volume);
void set_key_on(uint8_t channel, bool key_on);

#endif /* __MOD_H */