/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * 000.c
 *
 * 000: Magic Music Module PRO
 */

#include <string.h>

#include "../player.h"
#include "000.h"

bool load(const char *file_name)
{
    g_adpcm_parameters.sample_predictor = 0x8000;
    g_adpcm_parameters.delta_n = 0x007F;

    g_roboplay_interface->open(file_name, false);

    g_roboplay_interface->read(&g_mmm_header, sizeof(MMM_HEADER));
    g_roboplay_interface->read((void *)DATA_SEGMENT_BASE, 0x3F00 - sizeof(MMM_HEADER));
    g_roboplay_interface->read(&g_instruments_during_play, sizeof(g_instruments_during_play));
    g_roboplay_interface->read((void *)DATA_SEGMENT_BASE, 0x36);
    g_roboplay_interface->read(&g_track_length, 1);
    g_roboplay_interface->read(&g_start_tempo, 1);

    g_roboplay_interface->close();

    strncpy(g_song_name, g_mmm_header.song_name, SONG_NAME_LENGTH);
    g_song_name[SONG_NAME_LENGTH] = '\0';

    strncpy(g_author, g_mmm_header.author, AUTHOR_NAME_LENGTH);
    g_author[AUTHOR_NAME_LENGTH] = '\0';

    /* Set OPL4 to memory access mode */
    g_roboplay_interface->opl_write_wave(0x02, 0x11);

    /* Set wave memory adress */
    uint32_t wave_address = OPL_WAVE_ADDRESS;
    g_roboplay_interface->opl_write_wave(3, (wave_address >> 16) & 0x3F);
    g_roboplay_interface->opl_write_wave(4, (wave_address >> 8)  & 0xFF);
    g_roboplay_interface->opl_write_wave(5, wave_address & 0xFF);

    /* Load the sample kit when available */
    g_play_samples = load_sample_kit(file_name, ".SM1");
    g_play_samples &= load_sample_kit(file_name, ".SM2");

    if(g_play_samples)
        write_sample_headers();

    /* Set OPL4 to sound generation mode */
    g_roboplay_interface->opl_write_wave(0x02, 0x10);

    /* Load the song data */
    char    *extension = "001";
    
    g_current_segment = 0;
    g_song_data_segments[g_current_segment] = START_SEGMENT_INDEX;
    while(load_song_data(file_name, extension) && g_current_segment < MAX_SONG_DATA_SEGMENTS)
    {
        extension[2]++;
        g_song_data_segments[++g_current_segment] = g_roboplay_interface->get_new_segment();
        g_roboplay_interface->set_segment(g_song_data_segments[g_current_segment]);
    }

    return true;
}

bool update()
{
#ifdef __DEBUG
    g_roboplay_interface->debug_string("\n\rupdate");
#endif

    g_tempo_count++;
    if(g_tempo_count > g_tempo)
    {
        g_tempo_count = 0;

        /* Play next step line */
        for(uint8_t i = 0; i < NR_OF_CHANNELS; i++)
        {
            uint8_t data = *g_pattern_data++;

#ifdef __DEBUG
    g_roboplay_interface->debug_string("\n\r");
    g_roboplay_interface->debug_hex(i);
    g_roboplay_interface->debug_string(":  ");
    g_roboplay_interface->debug_hex(data);
    g_roboplay_interface->debug_string(" ");
#endif

            {
                if(!data);
                else if(data < UNKNOWN) note_on(i, data);
                else if(data == UNKNOWN);
                else if(data == NOTE_OFF) note_off(i);
                // else if(data < FADE_IN) fade_out(data);
                // else if(data < VOLUME_CHANGE) fade_in(data);
                // else if(data < INSTRUMENT_CHANGE) change_volume(i, data);
                // else if(data < DETUNE_UP) change_instrument(i, data);
                // else if(data == DETUNE_UP) detune_up(i);
                // else if(data == DETUNE_DOWN) detune_down(i);
                // else if(data < MODULATE) change_tempo(data);
                // else if(data < TREMOLO) modulate(i);
                // else if(data < END_OF_TRACK) tremolo(i);
                // else if(data == EOT) end_of_track();
                // else if(data < PITCH_BEND_DOWN) pitch_bend_up(i, data);
                // else if(data < END_OF_FADE) pitch_bend_down(i, data);
                // else if(data == END_OF_FADE) stop_fade();
            }
        }

        if(g_play_samples && *g_pattern_data)
        {
            uint8_t frequency = *g_pattern_data++;
            uint8_t volume = *g_pattern_data++;
            uint8_t sample_number = *g_pattern_data++;

//            play_sample(frequency, volume, sample_number);
        }
        else
        {
            // Skip empty sample data
            g_pattern_data += 3;
        }

        g_current_line++;
        if(g_current_line >= g_track_length) next_track();
    }
    else
    {
        handle_pitch_bend();
    }

    return true;
}


void rewind(int8_t subsong)
{
    /* No subsongs in this format */
    subsong;

    for(uint8_t i = 0; i < NR_OF_CHANNELS; i++)
    {
        g_channel_status[i].note_active = false;
        g_channel_status[i].note_frequency.low  = 0;
        g_channel_status[i].note_frequency.high = 0;

        g_channel_status[i].pitch_bend_value_up   = 0;
        g_channel_status[i].pitch_bend_value_down = 0;
        g_channel_status[i].detune_value          = 0;

        change_instrument(i, g_mmm_header.start_instrument[i]);
    }

    g_tempo = g_start_tempo;
    g_tempo_count = 0;

    g_tracks_per_page = DATA_SEGMENT_SIZE / (sizeof(PATTERN_DATA) * g_track_length);
    g_track_data_size = g_track_length * LINE_DATA_LENGTH;

    g_pattern_index = 0;
    g_play_from_track  = g_mmm_header.pattern_table[g_pattern_index].play_from_track;
    g_play_until_track = g_mmm_header.pattern_table[g_pattern_index].play_until_track;
    g_repeat_count     = g_mmm_header.pattern_table[g_pattern_index].repeat_count;

    g_current_track = g_play_from_track;

    g_current_segment = g_play_from_track / g_tracks_per_page;
    g_roboplay_interface->set_segment(g_song_data_segments[g_current_segment]);

    g_pattern_data = (uint8_t *)(DATA_SEGMENT_BASE + ((g_play_from_track % g_tracks_per_page) * g_track_data_size));
    g_current_line    = 0;
}

void command(const uint8_t id)
{
    /* No additional commmands supported */
    id;
}

float get_refresh()
{
    /* Fixed replay rate of 60Hz */
    return 60.0;
}

uint8_t get_subsongs()
{
    return 0;
}

char* get_player_info()
{
    return "Magic Music Module PRO player by RoboSoft Inc.";
}

char* get_title()
{
    return g_song_name;
}

char* get_author()
{
    return g_author;
}

char* get_description()
{
    return "-";
}

bool load_song_data(const char *file_name, const char *extension)
{
    bool result = false;

    char* song_data_name = (char *)READ_BUFFER;

    strcpy(song_data_name, file_name);
    uint8_t found = strlen(song_data_name) - 1;
    while(song_data_name[found] != '.') found--;
    song_data_name[found+1] = '\0';

    strcat(song_data_name, extension);

    if(g_roboplay_interface->exists(song_data_name))
    {
        g_roboplay_interface->open(song_data_name, false);
        g_roboplay_interface->read((void *)DATA_SEGMENT_BASE, DATA_SEGMENT_SIZE);
        g_roboplay_interface->close();
        result = true;
    }

    return   result;
}

void write_sample_headers()
{
    uint16_t header_address = 0;

    for(int i = 0; i < NR_OF_SAMPLE_BLOCKS; i++)
    {
        uint32_t start = OPL_WAVE_ADDRESS + g_sample_blocks[i].start_address;
        uint16_t loop = g_sample_blocks[i].end_address - 1;
        uint16_t end = (g_sample_blocks[i].end_address - 1) ^ 0xFFFF;

        /* Write header to OPL memory */
        g_roboplay_interface->opl_write_wave(3, 0x20);
        g_roboplay_interface->opl_write_wave(4, (header_address >> 8) & 0xFF);
        g_roboplay_interface->opl_write_wave(5, header_address & 0xFF);

        /* Start address */
        g_roboplay_interface->opl_write_wave(6, (start >> 16) & 0x3F);
        g_roboplay_interface->opl_write_wave(6, (start >> 8) & 0xFF);
        g_roboplay_interface->opl_write_wave(6, start & 0xFF);

        /* Loop address */
        g_roboplay_interface->opl_write_wave(6, (loop >> 8) & 0xFF);
        g_roboplay_interface->opl_write_wave(6, loop & 0xFF);

        /* End address */
        g_roboplay_interface->opl_write_wave(6, (end >> 8) & 0xFF);
        g_roboplay_interface->opl_write_wave(6, end & 0xFF);


        g_roboplay_interface->opl_write_wave(6, 0);          /* LFO, VIB                */
        g_roboplay_interface->opl_write_wave(6, 0xF0);       /* AR, D1R                 */
        g_roboplay_interface->opl_write_wave(6, 0xFF);       /* DL, D2R                 */
        g_roboplay_interface->opl_write_wave(6, 0x0F);       /* Rate correction , RR    */
        g_roboplay_interface->opl_write_wave(6, 0);          /* AM                      */

        header_address += 12;
    }
}

bool load_sample_kit(const char* file_name, const char* extension)
{
    bool result = false;

    char* sample_kit_name = (char *)READ_BUFFER;

    strcpy(sample_kit_name, file_name);
    uint8_t found = strlen(sample_kit_name) - 1;
    while(sample_kit_name[found] != '\\' && sample_kit_name[found] != ':' && found > 0) found--;
    if(found) 
        sample_kit_name[found + 1] = '\0';
    else
        sample_kit_name[found] = '\0';

    uint8_t l = strlen(sample_kit_name);
    uint8_t c = 0;

    while(g_mmm_header.sample_kit_name[c] != ' ' && c < SAMPLE_KIT_NAME_LENGTH)
        sample_kit_name[l + c] = g_mmm_header.sample_kit_name[c++];
    sample_kit_name[l + c] = '\0';

    strcat(sample_kit_name, extension);
 
    if(g_roboplay_interface->exists(sample_kit_name))
    {
        result = true;

        uint8_t load_segment = g_roboplay_interface->get_new_segment();
        g_roboplay_interface->set_segment(load_segment);

        g_roboplay_interface->open(sample_kit_name, false);
        g_roboplay_interface->read((void *)READ_BUFFER, 7);

        for(int i = 0; i < 4; i++)
        {
            g_roboplay_interface->read((uint8_t *)READ_BUFFER, READ_BUFFER_SIZE);
            adpcm_to_pcm((uint8_t *)DATA_SEGMENT_BASE, (uint8_t *)READ_BUFFER, READ_BUFFER_SIZE);
            g_roboplay_interface->opl_write_wave_data((uint8_t*)DATA_SEGMENT_BASE, 2 * READ_BUFFER_SIZE);
        }
        g_roboplay_interface->close();

        g_roboplay_interface->set_segment(START_SEGMENT_INDEX);
    }

    return result;
}

void play_sample(uint8_t frequency, uint8_t volume, uint8_t sample_number)
{
    volume = 128 - (volume / 2);
    g_roboplay_interface->opl_write_wave(0x50, (volume << 1) | 0x01);

    uint16_t pitch_index = sizeof(g_pitch_table) - (22 * (frequency - 1));
    g_roboplay_interface->opl_write_wave(0x38, g_pitch_table[pitch_index + 1]);
    g_roboplay_interface->opl_write_wave(0x20, g_pitch_table[pitch_index] | 0x01);

    g_roboplay_interface->opl_write_wave(0x68, 0x00);

    g_roboplay_interface->opl_write_wave(0x08, 0x7F + sample_number);
    g_roboplay_interface->opl_wait_for_load();

    g_roboplay_interface->opl_write_wave(0x68, 0x80);
}

void next_track()
{

}

void frequency_up(uint8_t channel, uint8_t value)
{
    if(!(g_channel_status[channel].note_frequency.high & 0x01))
    {
        g_channel_status[channel].note_frequency.low += value;
        if(g_channel_status[channel].note_frequency.low > 0xB2)
        {
            g_channel_status[channel].note_frequency.low -= 0xB2;
            g_channel_status[channel].note_frequency.low += 0x59;
            g_channel_status[channel].note_frequency.high += 0x03;
        }
    }
    else
    {
        if(g_channel_status[channel].note_frequency.low + value > 255)
            g_channel_status[channel].note_frequency.high += 0x01;

        g_channel_status[channel].note_frequency.low += value;
    }

    g_roboplay_interface->opl_write_fm_1(0xA0 + channel, g_channel_status[channel].note_frequency.low);
    g_roboplay_interface->opl_write_fm_1(0xB0 + channel, g_channel_status[channel].note_frequency.high | 0x20);

    g_roboplay_interface->opl_write_fm_2(0xA0 + channel, g_channel_status[channel].note_frequency.low - 1);
    g_roboplay_interface->opl_write_fm_2(0xB0 + channel, g_channel_status[channel].note_frequency.high | 0x20);
}

void frequency_down(uint8_t channel, uint8_t value)
{
    if(g_channel_status[channel].note_frequency.high & 0x01)
    {
        g_channel_status[channel].note_frequency.low -= value;
        if(g_channel_status[channel].note_frequency.low < 0x46)
        {
            g_channel_status[channel].note_frequency.low -= 0x45;
            g_channel_status[channel].note_frequency.low += 0x8B;
            g_channel_status[channel].note_frequency.high -= 0x03;
        }
    }
    else
    {
        if(g_channel_status[channel].note_frequency.low < value)
            g_channel_status[channel].note_frequency.high -= 0x01;

        g_channel_status[channel].note_frequency.low -= value;
    }

    g_roboplay_interface->opl_write_fm_1(0xA0 + channel, g_channel_status[channel].note_frequency.low);
    g_roboplay_interface->opl_write_fm_1(0xB0 + channel, g_channel_status[channel].note_frequency.high | 0x20);

    g_roboplay_interface->opl_write_fm_2(0xA0 + channel, g_channel_status[channel].note_frequency.low - 1);
    g_roboplay_interface->opl_write_fm_2(0xB0 + channel, g_channel_status[channel].note_frequency.high | 0x20);
}

void handle_pitch_bend()
{
    for(uint8_t i = 0; i < NR_OF_CHANNELS; i++)
    {
        if(g_channel_status[i].pitch_bend_value_up)
            frequency_up(i, g_channel_status[i].pitch_bend_value_up);

        if(g_channel_status[i].pitch_bend_value_down)
            frequency_down(i, g_channel_status[i].pitch_bend_value_down);
    }
}

void note_on(uint8_t channel, uint8_t data)
{
    g_channel_status[channel].note_active = true;
    g_channel_status[channel].pitch_bend_value_up = 0;
    g_channel_status[channel].pitch_bend_value_down = 0;
    g_channel_status[channel].detune_value = 0;

    g_channel_status[channel].note_frequency.low  = g_frequency_table[data - 1].low;
    g_channel_status[channel].note_frequency.high = g_frequency_table[data - 1].high;

    g_roboplay_interface->opl_write_fm_1(0xA0 + channel, g_channel_status[channel].note_frequency.low);

    g_roboplay_interface->opl_write_fm_1(0xB0 + channel, g_channel_status[channel].note_frequency.high);
    g_roboplay_interface->opl_write_fm_1(0xB0 + channel, g_channel_status[channel].note_frequency.high | 0x20);

    g_roboplay_interface->opl_write_fm_2(0xA0 + channel, g_channel_status[channel].note_frequency.low - 1);

    g_roboplay_interface->opl_write_fm_2(0xB0 + channel, g_channel_status[channel].note_frequency.high);
    g_roboplay_interface->opl_write_fm_2(0xB0 + channel, g_channel_status[channel].note_frequency.high | 0x20);
}

void note_off(uint8_t channel)
{
    g_channel_status[channel].note_active = false;

    g_roboplay_interface->opl_write_fm_1(0xB0 + channel, 0x00);
    g_roboplay_interface->opl_write_fm_2(0xB0 + channel, 0x00);
}

void fade_out(uint8_t data)
{

}

void fade_in(uint8_t data)
{

}

void stop_fade()
{

}

void modulate(uint8_t channel)
{

}

void tremolo(uint8_t channel)
{

}

void detune_up(uint8_t channel)
{
    if(g_channel_status[channel].note_active)
    {
        frequency_up(channel, 1);
    }
}

void detune_down(uint8_t channel)
{
    if(g_channel_status[channel].note_active)
    {
        frequency_down(channel, 1);
    }
}

void pitch_bend_up(uint8_t channel, uint8_t data)
{
    if(g_channel_status[channel].note_active)
    {
        uint8_t value = data - PITCH_BEND_UP;
        g_channel_status[channel].pitch_bend_value_up   = value;
        g_channel_status[channel].pitch_bend_value_down = 0;

        frequency_up(channel, g_channel_status[channel].pitch_bend_value_up);
    }
}

void pitch_bend_down(uint8_t channel, uint8_t data)
{
    if(g_channel_status[channel].note_active)
    {
        int8_t value = data - PITCH_BEND_DOWN;
        g_channel_status[channel].pitch_bend_value_up   = 0;
        g_channel_status[channel].pitch_bend_value_down = value;

        frequency_down(channel, g_channel_status[channel].pitch_bend_value_down);
    }
}

void change_tempo(uint8_t data)
{
    uint8_t value = data - TEMPO_CHANGE;
    g_tempo = value;
}

void change_volume(uint8_t channel, uint8_t data)
{

}

void change_instrument(uint8_t channel, uint8_t nr)
{
    for(uint8_t i = 0; i < INSTRUMENT_DATA_SIZE - 1; i++)
    {
        g_roboplay_interface->opl_write_fm_1(g_instrument_registers[channel][i], g_instruments_during_play[nr].data[i]);
    }
    g_roboplay_interface->opl_write_fm_1(
        g_instrument_registers[channel][INSTRUMENT_DATA_SIZE - 1], 
        g_instruments_during_play[nr].data[INSTRUMENT_DATA_SIZE - 1] | PAN_SETTING_MID);
}

void end_of_track()
{

}

void adpcm_to_pcm(uint8_t *destination, uint8_t *source, uint16_t size)
{
    while(size--)
    {
        int8_t adpcm_value = *source++;
        *destination++ = adpcm_code_to_pcm_code(adpcm_value >> 4) + 128;
        *destination++ = adpcm_code_to_pcm_code(adpcm_value) + 128;
    }
}

uint8_t adpcm_code_to_pcm_code(int8_t adpcm_value)
{
    uint16_t old_delta_n = g_adpcm_parameters.delta_n;
    uint16_t wrk_delta_n = old_delta_n;

    wrk_delta_n >>= 1;
    if(adpcm_value & 1)
        wrk_delta_n += old_delta_n;

    wrk_delta_n >>= 1;
    if(adpcm_value & 2)
        wrk_delta_n += old_delta_n;

    wrk_delta_n >>= 1;
    if(adpcm_value & 4)
        wrk_delta_n += old_delta_n;

    if(adpcm_value & 8)
        if(g_adpcm_parameters.sample_predictor > wrk_delta_n)
            g_adpcm_parameters.sample_predictor -= wrk_delta_n;
        else
            g_adpcm_parameters.sample_predictor = 0;
    else
        if(0xFFFF - wrk_delta_n > g_adpcm_parameters.sample_predictor)
            g_adpcm_parameters.sample_predictor += wrk_delta_n;
        else
            g_adpcm_parameters.sample_predictor = 0xFFFF;

    static const uint8_t factors[] = {0x39, 0x39, 0x39, 0x39, 0x4D, 0x66, 0x80, 0x99};

    uint8_t factor = factors[adpcm_value & 7];
    wrk_delta_n = 0;
    for(uint8_t i = 6; i > 0; i--) 
    {
        if(factor & 1)
            wrk_delta_n += old_delta_n;
        wrk_delta_n >>= 1;
        factor >>= 1;
    }
    if(factor & 1)
        wrk_delta_n += old_delta_n;
    factor >>= 1;
    if(factor & 1)
        wrk_delta_n += (old_delta_n << 1);

    /* Clip delta_n against the upper/lower bounds */
    if(wrk_delta_n > 0x6000)
        g_adpcm_parameters.delta_n = 0x6000;
    else
        if (wrk_delta_n < 127)
            g_adpcm_parameters.delta_n = 127;
        else
          g_adpcm_parameters.delta_n = wrk_delta_n;

  /* PCM value is predicted pcm value / 256 */
  return g_adpcm_parameters.sample_predictor >> 8;
}
