/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * 000.h
 *
 * 000: Magic Music Module PRO
 */

#ifndef __000_H
#define __000_H

#include <stdint.h>
#include <stdbool.h>

#define OPL_WAVE_MEMORY  0x200000
#define OPL_WAVE_ADDRESS OPL_WAVE_MEMORY + 384 * 12

#define NR_OF_SAMPLE_BLOCKS 15
#define NR_OF_SAMPLE_PITCHES 60

#define SONG_NAME_LENGTH       40
#define AUTHOR_NAME_LENGTH     40
#define SAMPLE_KIT_NAME_LENGTH 8

#define MAX_SONG_DATA_SEGMENTS 8

#define LINE_DATA_LENGTH 9 + 3  /* Nine channels, frequentie, volume, sample */

#define NR_OF_PATTERNS 20
#define NR_OF_INSTRUMENTS 128
#define NR_OF_INSTRUMENTS_DURING_PLAY 20

#define NR_OF_CHANNELS 9
#define NR_OF_DRUM_CHANNELS 3

#define FIRST_DRUM_CHANNEL  6

#define BRIGHTNESS_REGISTER_INDEX 2
#define VOLUME_REGISTER_INDEX     3

#define SAMPLE_DATA_SIZE 3
#define STEP_DATA_SIZE   12
#define PATTERN_SIZE     16

#define MAX_SONG_SIZE 0x3F00

#define MAX_TEMPO 24

#define INSTRUMENT_DATA_SIZE 9

#define UNKNOWN           92
#define NOTE_OFF          93
#define FADE_OUT          94
#define FADE_IN           98
#define VOLUME_CHANGE     102
#define INSTRUMENT_CHANGE 166
#define DETUNE_UP         186
#define DETUNE_DOWN       187
#define TEMPO_CHANGE      188
#define MODULATE          213
#define TREMOLO           214
#define END_OF_TRACK      215
#define PITCH_BEND_UP     216
#define PITCH_BEND_DOWN   226
#define END_OF_SONG       236

#define PAN_SETTING_LEFT  0x10
#define PAN_SETTING_MID   0x30
#define PAN_SETTING_RIGHT 0x20

typedef struct
{
    uint16_t play_from_track;
    uint16_t play_until_track;
    uint8_t repeat_count;
} PATTERN_DATA;

typedef struct
{
    char name[20];
    uint8_t data[9];
} INSTRUMENT_DATA;

typedef struct
{
    uint8_t data[9];
    uint8_t instrument_number;
} INSTRUMENT_DURING_PLAY;

typedef struct
{
    uint8_t  start_volume[NR_OF_CHANNELS];
    uint8_t  start_instrument[NR_OF_CHANNELS];
    char     song_name[SONG_NAME_LENGTH];
    char     author[AUTHOR_NAME_LENGTH];

    uint8_t  nr_of_mapper_pages;

    PATTERN_DATA pattern_table[NR_OF_PATTERNS];
    uint16_t pattern_end;    

    char     sample_kit_name[SAMPLE_KIT_NAME_LENGTH];
    uint8_t  transpose_table[NR_OF_CHANNELS];
    bool     repeat;  
} MMM_HEADER;

typedef struct 
{
    uint8_t low;
    uint8_t high;
} MMM_FREQUENCY;

typedef struct
{
    bool          note_active;
    MMM_FREQUENCY note_frequency;

    uint8_t       pitch_bend_value_up;
    uint8_t       pitch_bend_value_down;
    int8_t        detune_value;

    uint8_t       volume;
} MMM_STATUS_TABLE;

typedef struct
{
    uint16_t sample_predictor;
    uint16_t delta_n;
} ADPCM_PARAMETERS;

typedef struct
{
    uint16_t start_address;
    uint16_t end_address;
} SAMPLE_BLOCK;

typedef struct
{
    uint8_t low;
    uint8_t high;
} SAMPLE_PITCH;

static const MMM_FREQUENCY g_frequency_table[] =
    {
        {0xAE, 0x22}, {0xD7, 0x22}, {0x02, 0x23}, {0x30, 0x23}, {0x60, 0x23}, {0x94, 0x23},
        {0xCA, 0x23}, {0x02, 0x26}, {0x20, 0x26}, {0x41, 0x26}, {0x63, 0x26}, {0x87, 0x26},
        {0xAE, 0x26}, {0xD7, 0x26}, {0x02, 0x27}, {0x30, 0x27}, {0x60, 0x27}, {0x94, 0x27},
        {0xCA, 0x27}, {0x02, 0x2A}, {0x20, 0x2A}, {0x41, 0x2A}, {0x63, 0x2A}, {0x87, 0x2A},
        {0xAE, 0x2A}, {0xD7, 0x2A}, {0x02, 0x2B}, {0x30, 0x2B}, {0x60, 0x2B}, {0x94, 0x2B},
        {0xCA, 0x2B}, {0x02, 0x2E}, {0x20, 0x2E}, {0x41, 0x2E}, {0x63, 0x2E}, {0x87, 0x2E},
        {0xAE, 0x2E}, {0xD7, 0x2E}, {0x02, 0x2F}, {0x30, 0x2F}, {0x60, 0x2F}, {0x94, 0x2F},
        {0xCA, 0x2F}, {0x02, 0x32}, {0x20, 0x32}, {0x41, 0x32}, {0x63, 0x32}, {0x87, 0x32},
        {0xAE, 0x32}, {0xD7, 0x32}, {0x02, 0x33}, {0x30, 0x33}, {0x60, 0x33}, {0x94, 0x33},
        {0xCA, 0x33}, {0x02, 0x36}, {0x20, 0x36}, {0x41, 0x36}, {0x63, 0x36}, {0x87, 0x36},
        {0xAE, 0x36}, {0xD7, 0x36}, {0x02, 0x37}, {0x30, 0x37}, {0x60, 0x37}, {0x94, 0x37},
        {0xCA, 0x37}, {0x02, 0x3A}, {0x20, 0x3A}, {0x41, 0x3A}, {0x63, 0x3A}, {0x87, 0x3A},
        {0xAE, 0x3A}, {0xD7, 0x3A}, {0x02, 0x3B}, {0x30, 0x3B}, {0x60, 0x3B}, {0x94, 0x3B},
        {0xCA, 0x3B}, {0x02, 0x3E}, {0x20, 0x3E}, {0x41, 0x3E}, {0x63, 0x3E}, {0x87, 0x3E},
        {0xAE, 0x3E}, {0xD7, 0x3E}, {0x02, 0x3F}, {0x30, 0x3F}, {0x60, 0x3F}, {0x94, 0x3F},
        {0xCA, 0x3F}
    };

static const uint8_t g_instrument_registers[NR_OF_CHANNELS][INSTRUMENT_DATA_SIZE] =
    {
        {0x83, 0x80, 0x63, 0x60, 0x43, 0x40, 0x23, 0x20, 0xC0},
        {0x84, 0x81, 0x64, 0x61, 0x44, 0x41, 0x24, 0x21, 0xC1},
        {0x85, 0x82, 0x65, 0x62, 0x45, 0x42, 0x25, 0x22, 0xC2},

        {0x8B, 0x88, 0x6B, 0x68, 0x4B, 0x48, 0x2B, 0x28, 0xC3},
        {0x8C, 0x89, 0x6C, 0x69, 0x4C, 0x49, 0x2C, 0x29, 0xC4},
        {0x8D, 0x8A, 0x6D, 0x6A, 0x4D, 0x4A, 0x2D, 0x2A, 0xC5},

        {0x93, 0x90, 0x73, 0x70, 0x53, 0x50, 0x33, 0x30, 0xC6},
        {0x94, 0x91, 0x74, 0x71, 0x54, 0x51, 0x34, 0x31, 0xC7},
        {0x95, 0x92, 0x75, 0x72, 0x55, 0x52, 0x35, 0x32, 0xC8},
    };

static const SAMPLE_BLOCK g_sample_blocks[NR_OF_SAMPLE_BLOCKS] = 
    {
        {0x0000, 0x1FFF}, {0x2000, 0x1FFF}, {0x4000, 0x1FFF}, {0x6000, 0x1FFF},
        {0x8000, 0x1FFF}, {0xA000, 0x1FFF}, {0xC000, 0x1FFF}, {0xE000, 0x1FFF},
        {0x0000, 0x3FFF}, {0x4000, 0x3FFF}, {0x8000, 0x3FFF}, {0xC000, 0x3FFF},
        {0x0000, 0x7FFF}, {0x8000, 0x7FFF},
        {0x0000, 0xFFFF}
    };

static const uint8_t g_pitch_table[] =
{
    0x06,0x04,0xEA,0x03,0xCE,0x03,0xB2,0x03,0x98,0x03,0x7E,0x03,0x64,0x03,0x4A,0x03,0x32,0x03,0x18,0x03,0x00,0x03,0xE8,0x02,
    0xD2,0x02,0xBA,0x02,0xA4,0x02,0x8E,0x02,0x78,0x02,0x62,0x02,0x4E,0x02,0x38,0x02,0x24,0x02,0x10,0x02,0xFC,0x01,0xE8,0x01,
    0xD6,0x01,0xC2,0x01,0xB0,0x01,0x9E,0x01,0x8C,0x01,0x7A,0x01,0x68,0x01,0x56,0x01,0x46,0x01,0x34,0x01,0x24,0x01,0x14,0x01,
    0x04,0x01,0xF4,0x00,0xE4,0x00,0xD4,0x00,0xC6,0x00,0xB6,0x00,0xA8,0x00,0x98,0x00,0x8A,0x00,0x7C,0x00,0x6E,0x00,0x60,0x00,
    0x52,0x00,0x44,0x00,0x38,0x00,0x2A,0x00,0x1C,0x00,0x10,0x00,0x04,0x00,0xEE,0xF7,0xD6,0xF7,0xBC,0xF7,0xA4,0xF7,0x8C,0xF7,
    0x74,0xF7,0x5E,0xF7,0x46,0xF7,0x30,0xF7,0x18,0xF7,0x02,0xF7,0xEC,0xF6,0xD6,0xF6,0xC2,0xF6,0xAC,0xF6,0x96,0xF6,0x82,0xF6,
    0x6E,0xF6,0x58,0xF6,0x44,0xF6,0x30,0xF6,0x1C,0xF6,0x0A,0xF6,0xF6,0xF5,0xE2,0xF5,0xD0,0xF5,0xBE,0xF5,0xAA,0xF5,0x98,0xF5,
    0x86,0xF5,0x74,0xF5,0x62,0xF5,0x50,0xF5,0x40,0xF5,0x2E,0xF5,0x1E,0xF5,0x0C,0xF5,0xFC,0xF4,0xEC,0xF4,0xDA,0xF4,0xCA,0xF4,
    0xBA,0xF4,0xAA,0xF4,0x9A,0xF4,0x8C,0xF4,0x7C,0xF4,0x6C,0xF4,0x5E,0xF4,0x4E,0xF4,0x40,0xF4,0x30,0xF4,0x22,0xF4,0x14,0xF4,
    0x06,0xF4,0xF8,0xF3,0xEA,0xF3,0xDC,0xF3,0xCE,0xF3,0xC0,0xF3,0xB2,0xF3,0xA4,0xF3,0x98,0xF3,0x8A,0xF3,0x7E,0xF3,0x70,0xF3,
    0x64,0xF3,0x56,0xF3,0x4A,0xF3,0x3E,0xF3,0x32,0xF3,0x24,0xF3,0x18,0xF3,0x0C,0xF3,0x00,0xF3,0xF4,0xF2,0xE8,0xF2,0xDE,0xF2,
    0xD2,0xF2,0xC6,0xF2,0xBA,0xF2,0xB0,0xF2,0xA4,0xF2,0x9A,0xF2,0x8E,0xF2,0x84,0xF2,0x78,0xF2,0x6E,0xF2,0x62,0xF2,0x58,0xF2,
    0x4E,0xF2,0x44,0xF2,0x38,0xF2,0x2E,0xF2,0x24,0xF2,0x1A,0xF2,0x10,0xF2,0x06,0xF2,0xFC,0xF1,0xF2,0xF1,0xE8,0xF1,0xE0,0xF1,
    0xD6,0xF1,0xCC,0xF1,0xC2,0xF1,0xBA,0xF1,0xB0,0xF1,0xA6,0xF1,0x9E,0xF1,0x94,0xF1,0x8C,0xF1,0x82,0xF1,0x7A,0xF1,0x70,0xF1,
    0x68,0xF1,0x60,0xF1,0x56,0xF1,0x4E,0xF1,0x46,0xF1,0x3E,0xF1,0x34,0xF1,0x2C,0xF1,0x24,0xF1,0x1C,0xF1,0x14,0xF1,0x0C,0xF1,
    0x04,0xF1,0xFC,0xF0,0xF4,0xF0,0xEC,0xF0,0xE4,0xF0,0xDC,0xF0,0xD4,0xF0,0xCE,0xF0,0xC6,0xF0,0xBE,0xF0,0xB6,0xF0,0xAE,0xF0,
    0xA8,0xF0,0xA0,0xF0,0x98,0xF0,0x92,0xF0,0x8A,0xF0,0x84,0xF0,0x7C,0xF0,0x74,0xF0,0x6E,0xF0,0x66,0xF0,0x60,0xF0,0x5A,0xF0,
    0x52,0xF0,0x4C,0xF0,0x44,0xF0,0x3E,0xF0,0x38,0xF0,0x30,0xF0,0x2A,0xF0,0x24,0xF0,0x1C,0xF0,0x16,0xF0,0x10,0xF0,0x0A,0xF0,
    0x04,0xF0,0xFA,0xE7,0xEE,0xE7,0xE2,0xE7,0xD6,0xE7,0xCA,0xE7,0xBC,0xE7,0xB0,0xE7,0xA4,0xE7,0x98,0xE7,0x8C,0xE7,0x80,0xE7,
    0x74,0xE7,0x6A,0xE7,0x5E,0xE7,0x52,0xE7,0x46,0xE7,0x3A,0xE7,0x30,0xE7,0x24,0xE7,0x18,0xE7,0x0E,0xE7,0x02,0xE7,0xF8,0xE6,
    0xEC,0xE6,0xE2,0xE6,0xD6,0xE6,0xCC,0xE6,0xC2,0xE6,0xB6,0xE6,0xAC,0xE6,0xA2,0xE6,0x96,0xE6,0x8c,0xe6,0x82,0xe6,0x78,0xe6,
    0x6e,0xe6,0x62,0xe6,0x58,0xe6,0x4e,0xe6,0x44,0xE6,0x3A,0xE6,0x30,0xE6,0x26,0xE6,0x1C,0xE6,0x12,0xE6,0x0A,0xE6,0x00,0xE6,
    0xF6,0xE5,0xEC,0xE5,0xE2,0xE5,0xDA,0xE5,0xD0,0xE5,0xC6,0xE5,0xBE,0xE5,0xB4,0xE5,0xAA,0xE5,0xA2,0xE5,0x98,0xE5,0x90,0xE5,
    0x86,0xE5,0x7E,0xE5,0x74,0xE5,0x6C,0xE5,0x62,0xE5,0x5A,0xE5,0x50,0xE5,0x48,0xE5,0x40,0xE5,0x36,0xE5,0x2E,0xE5,0x26,0xE5,
    0x1E,0xE5,0x14,0xE5,0x0C,0xE5,0x04,0xE5,0xFC,0xE4,0xF4,0xE4,0xEC,0xE4,0xE2,0xE4,0xDA,0xE4,0xD2,0xE4,0xCA,0xE4,0xC2,0xE4,
    0xBA,0xE4,0xB2,0xE4,0xAA,0xE4,0xA2,0xE4,0x9A,0xE4,0x94,0xE4,0x8C,0xE4,0x84,0xE4,0x7C,0xE4,0x74,0xE4,0x6C,0xE4,0x64,0xE4,
    0x5E,0xE4,0x56,0xE4,0x4E,0xE4,0x46,0xE4,0x40,0xE4,0x38,0xE4,0x30,0xE4,0x2A,0xE4,0x22,0xE4,0x1A,0xE4,0x14,0xE4,0x0C,0xE4,
    0x06,0xE4,0xFE,0xE3,0xF8,0xE3,0xF0,0xE3,0xEA,0xE3,0xE2,0xE3,0xDC,0xE3,0xD4,0xE3,0xCE,0xE3,0xC6,0xE3,0xC0,0xE3,0xB8,0xE3,
    0xB2,0xE3,0xAC,0xE3,0xA4,0xE3,0x9E,0xE3,0x98,0xE3,0x90,0xE3,0x8A,0xE3,0x84,0xE3,0x7E,0xE3,0x76,0xE3,0x70,0xE3,0x6A,0xE3,
    0x64,0xE3,0x5E,0xE3,0x56,0xE3,0x50,0xE3,0x4A,0xE3,0x44,0xE3,0x3E,0xE3,0x38,0xE3,0x32,0xE3,0x2A,0xE3,0x24,0xE3,0x1E,0xE3,
    0x18,0xE3,0x12,0xE3,0x0C,0xE3,0x06,0xE3,0x00,0xE3,0xFA,0xE2,0xF4,0xE2,0xEE,0xE2,0xE8,0xE2,0xE2,0xE2,0xDE,0xE2,0xD8,0xE2,
    0xD2,0xE2,0xCC,0xE2,0xC6,0xE2,0xC0,0xE2,0xBA,0xE2,0xB6,0xE2,0xB0,0xE2,0xAA,0xE2,0xA4,0xE2,0x9E,0xE2,0x9A,0xE2,0x94,0xE2,
    0x8E,0xE2,0x88,0xE2,0x84,0xE2,0x7E,0xE2,0x78,0xE2,0x72,0xE2,0x6E,0xE2,0x68,0xE2,0x62,0xE2,0x5E,0xE2,0x58,0xE2,0x52,0xE2,
    0x4E,0xE2,0x48,0xE2,0x44,0xE2,0x3E,0xE2,0x38,0xE2,0x34,0xE2,0x2E,0xE2,0x2A,0xE2,0x24,0xE2,0x20,0xE2,0x1A,0xE2,0x16,0xE2,
    0x10,0xE2,0x0C,0xE2,0x06,0xE2,0x02,0xE2,0xFC,0xE1,0xF8,0xE1,0xF2,0xE1,0xEE,0xE1,0xE8,0xE1,0xE4,0xE1,0xE0,0xE1,0xDA,0xE1,
    0xD6,0xE1,0xD0,0xE1,0xCC,0xE1,0xC8,0xE1,0xC2,0xE1,0xBE,0xE1,0xBA,0xE1,0xB4,0xE1,0xB0,0xE1,0xAC,0xE1,0xA6,0xE1,0xA2,0xE1,
    0x9E,0xE1,0x9A,0xE1,0x94,0xE1,0x90,0xE1,0x8C,0xE1,0x88,0xE1,0x82,0xE1,0x7E,0xE1,0x7A,0xE1,0x76,0xE1,0x70,0xE1,0x6C,0xE1,
    0x68,0xE1,0x64,0xE1,0x60,0xE1,0x5C,0xE1,0x56,0xE1,0x52,0xE1,0x4E,0xE1,0x4A,0xE1,0x46,0xE1,0x42,0xE1,0x3E,0xE1,0x3A,0xE1,
    0x34,0xE1,0x30,0xE1,0x2C,0xE1,0x28,0xE1,0x24,0xE1,0x20,0xE1,0x1C,0xE1,0x18,0xE1,0x14,0xE1,0x10,0xE1,0x0C,0xE1,0x08,0xE1,
    0x04,0xE1,0x00,0xE1,0xFC,0xE0,0xF8,0xE0,0xF4,0xE0,0xF0,0xE0,0xEC,0xE0,0xE8,0xE0,0xE4,0xE0,0xE0,0xE0,0xDC,0xE0,0xD8,0xE0,
    0xD4,0xE0,0xD0,0xE0,0xCE,0xE0,0xCA,0xE0,0xC6,0xE0,0xC2,0xE0,0xBE,0xE0,0xBA,0xE0,0xB6,0xE0,0xB2,0xE0,0xAE,0xE0,0xAC,0xE0,
    0xA8,0xE0,0xA4,0xE0,0xA0,0xE0,0x9C,0xE0,0x98,0xE0,0x96,0xE0,0x92,0xE0,0x8E,0xE0,0x8A,0xE0,0x86,0xE0,0x84,0xE0,0x80,0xE0,
    0x7C,0xE0,0x78,0xE0,0x74,0xE0,0x72,0xE0,0x6E,0xE0,0x6A,0xE0,0x66,0xE0,0x64,0xE0,0x60,0xE0,0x5C,0xE0,0x5A,0xE0,0x56,0xE0,
    0x52,0xE0,0x4E,0xE0,0x4C,0xE0,0x48,0xE0,0x44,0xE0,0x42,0xE0,0x3E,0xE0,0x3A,0xE0,0x38,0xE0,0x34,0xE0,0x30,0xE0,0x2E,0xE0,
    0x2A,0xE0,0x26,0xE0,0x24,0xE0,0x20,0xE0,0x1C,0xE0,0x1A,0xE0,0x16,0xE0,0x14,0xE0,0x10,0xE0,0x0C,0xE0,0x0A,0xE0,0x06,0xE0,
    0x04,0xE0,0x00,0xE0,0xFA,0xD7,0xF4,0xD7,0xEE,0xD7,0xE8,0xD7,0xE2,0xD7,0xDC,0xD7,0xD6,0xD7,0xD0,0xD7,0xCA,0xD7,0xC4,0xD7,
    0xBC,0xD7,0xB6,0xD7,0xB0,0xD7,0xAA,0xD7,0xA4,0xD7,0x9E,0xD7,0x98,0xD7,0x92,0xD7,0x8C,0xD7,0x86,0xD7,0x80,0xD7,0x7A,0xD7,
    0x74,0xD7,0x70,0xD7,0x6A,0xD7,0x64,0xD7,0x5E,0xD7,0x58,0xD7,0x52,0xD7,0x4C,0xD7,0x46,0xD7,0x40,0xD7,0x3A,0xD7,0x36,0xD7,
    0x30,0xD7,0x2A,0xD7,0x24,0xD7,0x1E,0xD7,0x18,0xD7,0x14,0xD7,0x0E,0xD7,0x08,0xD7,0x02,0xD7,0xFE,0xD6,0xF8,0xD6,0xF2,0xD6,
    0xEC,0xD6,0xE6,0xD6,0xE2,0xD6,0xDC,0xD6,0xD6,0xD6,0xD2,0xD6,0xCC,0xD6,0xC6,0xD6,0xC2,0xD6,0xBC,0xD6,0xB6,0xD6,0xB2,0xD6,
    0xAC,0xD6,0xA6,0xD6,0xA2,0xD6,0x9C,0xD6,0x96,0xD6,0x92,0xD6,0x8C,0xD6,0x86,0xD6,0x82,0xD6,0x7C,0xD6,0x78,0xD6,0x72,0xD6,
    0x6E,0xD6,0x68,0xD6,0x62,0xD6,0x5E,0xD6,0x58,0xD6,0x54,0xD6,0x4E,0xD6,0x4A,0xD6,0x44,0xD6,0x40,0xD6,0x3A,0xD6,0x36,0xD6,
    0x30,0xD6,0x2C,0xD6,0x26,0xD6,0x22,0xD6,0x1C,0xD6,0x18,0xD6,0x12,0xD6,0x0E,0xD6,0x0A,0xD6,0x04,0xD6,0x00,0xD6,0xFA,0xD5,
    0xF6,0xD5,0xF2,0xD5,0xEC,0xD5,0xE8,0xD5,0xE2,0xD5,0xDE,0xD5,0xDA,0xD5,0xD4,0xD5,0xD0,0xD5,0xCC,0xD5,0xC6,0xD5,0xC2,0xD5,
    0xBE,0xD5,0xB8,0xD5,0xB4,0xD5,0xB0,0xD5,0xAA,0xD5,0xA6,0xD5,0xA2,0xD5,0x9C,0xD5,0x98,0xD5,0x94,0xD5,0x90,0xD5,0x8A,0xD5,
    0x86,0xD5,0x82,0xD5,0x7E,0xD5,0x78,0xD5,0x74,0xD5,0x70,0xD5,0x6C,0xD5,0x66,0xD5,0x62,0xD5,0x5E,0xD5,0x5A,0xD5,0x56,0xD5,
};

bool load_song_data(const char* file_name, const char* extension);

void write_sample_headers();
bool load_sample_kit(const char* file_name, const char* extension);

void play_sample(uint8_t frequency, uint8_t volume, uint8_t sample_number);

void next_track();

void frequency_up(uint8_t channel, uint8_t value);
void frequency_down(uint8_t channel, uint8_t value);
void handle_pitch_bend();

void note_on(uint8_t channel, uint8_t data);
void note_off(uint8_t channel);

void fade_out(uint8_t data);
void fade_in(uint8_t data);
void stop_fade();

void modulate(uint8_t channel);
void tremolo(uint8_t channel);

void detune_up(uint8_t channel);
void detune_down(uint8_t channel);
void pitch_bend_up(uint8_t channel, uint8_t data);
void pitch_bend_down(uint8_t channel, uint8_t data);

void change_tempo(uint8_t data);
void change_volume(uint8_t channel, uint8_t data);
void change_instrument(uint8_t channel, uint8_t data);

void end_of_track();

void adpcm_to_pcm(uint8_t *destination, uint8_t *source, uint16_t size);
uint8_t adpcm_code_to_pcm_code(int8_t adpcm_value);

static MMM_HEADER g_mmm_header;
static uint8_t  g_track_length;
static uint8_t  g_start_tempo;

static uint8_t  g_song_data_segments[MAX_SONG_DATA_SEGMENTS];
static uint8_t  g_current_segment;

static INSTRUMENT_DURING_PLAY g_instruments_during_play[NR_OF_INSTRUMENTS_DURING_PLAY];

static char g_song_name[SONG_NAME_LENGTH + 1];
static char g_author[AUTHOR_NAME_LENGTH + 1];

static MMM_STATUS_TABLE g_channel_status[NR_OF_CHANNELS];

static uint8_t g_tempo;
static uint8_t g_tempo_count;

static uint8_t *g_pattern_data;

static uint8_t  g_pattern_index;

static uint8_t  g_tracks_per_page;
static uint8_t  g_track_data_size;

static uint8_t  g_current_line;
static uint8_t  g_current_track;
static uint16_t g_play_from_track;
static uint16_t g_play_until_track;
static uint8_t  g_repeat_count;

static bool g_play_samples;

static ADPCM_PARAMETERS g_adpcm_parameters;

#endif /* __000_H */
