/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 *
 * mus.h
 *
 * MUS: FAC SoundTracker
 */

#pragma once

#include <stdint.h>
#include <stdbool.h>

#define OPL_WAVE_MEMORY  0x200000
#define OPL_WAVE_ADDRESS OPL_WAVE_MEMORY + 384 * 12

#define NR_OF_SAMPLE_BLOCKS 15
#define NR_OF_SAMPLE_PITCHES 60

#define SONG_NAME_LENGTH     40
#define AUTHOR_NAME_LENGTH   40

#define NR_OF_CHANNELS 9
#define NR_OF_DRUM_CHANNELS 3

#define FIRST_DRUM_CHANNEL  6

#define BRIGHTNESS_REGISTER_INDEX 2
#define VOLUME_REGISTER_INDEX     3

#define SAMPLE_DATA_SIZE 3
#define STEP_DATA_SIZE   12
#define PATTERN_SIZE     16

#define MAX_SONG_SIZE 0x3F00

#define MAX_TEMPO 0x72

#define INSTRUMENT_DATA_SIZE 11
#define PRESET_DATA_SIZE     11

#define MUSIC_NR_OF_PRESETS        16
#define MUSIC_INSTRUMENT_DATA_SIZE 8

#define NOTE_OFF          0x40
#define DETUNE_UP         0x41
#define DETUNE_DOWN       0x42
#define PITCH_BEND_UP     0x43
#define PITCH_BEND_DOWN   0x4D
#define TEMPO_CHANGE      0x57
#define VOLUME_CHANGE     0x80
#define BRIGHTNESS_CHANGE 0xC0

#define PAN_SETTING_LEFT  0x10
#define PAN_SETTING_MID   0x30
#define PAN_SETTING_RIGHT 0x20

#define DEVICE_AUDIO  0x00
#define DEVICE_MUSIC  0x01

typedef struct
{
    uint8_t  voice_data_audio[NR_OF_CHANNELS][INSTRUMENT_DATA_SIZE];
    uint8_t  instrument_list_audio[NR_OF_CHANNELS];
    uint8_t  original_data_music[8];
    uint8_t  instrument_volume_music[6];
    uint8_t  midi_drum[8];
    uint16_t speed;
    char     sample_kit_name[8];
    uint8_t  version;
    uint8_t  original_number_music;
    uint8_t  user_id;
    uint8_t  midi_register_number[NR_OF_CHANNELS];
    uint8_t  midi_id_byte[NR_OF_CHANNELS];
    uint8_t  midi_transpose_data[5];
    char     song_name[40];                     /* PRO: Length -5 */
    /* uint8_t  midi_velocity_Data[5]; */       /* PRO: First 5 channels */
    uint8_t  sustain_audio;
    char     author[40];                        /* PRO: Length -5 */
    /* uint8_t  midi_velocity_Data[5]; */       /* PRO:  Last 4 channels */
    uint8_t  device_id;
    uint8_t  number_of_tracks;
    uint8_t  midi_drum_id[7];
} MUS_HEADER;

typedef struct 
{
    uint8_t low;
    uint8_t high;
} MUS_FREQUENCY;

typedef struct
{
    bool          note_active;
    MUS_FREQUENCY note_frequency;

    uint8_t       pitch_bend_value_up;
    uint8_t       pitch_bend_value_down;
    int8_t        detune_value;
} MUS_STATUS_TABLE;

typedef struct
{
    uint16_t start_address;
    uint16_t end_address;
} SAMPLE_BLOCK;

typedef struct
{
    uint8_t low;
    uint8_t high;
} SAMPLE_PITCH;

static const MUS_FREQUENCY g_frequency_table[] =
    {
        {0x59, 0x09}, {0x6D, 0x09}, {0x83, 0x09}, {0x9A, 0x09}, {0xB2, 0x09}, {0xCC, 0x09}, {0xE8, 0x09}, {0x05, 0x0A}, {0x23, 0x0A}, {0x44, 0x0A}, {0x67, 0x0A}, {0x8B, 0x0A}, 
        {0x59, 0x0D}, {0x6D, 0x0D}, {0x83, 0x0D}, {0x9A, 0x0D}, {0xB2, 0x0D}, {0xCC, 0x0D}, {0xE8, 0x0D}, {0x05, 0x0E}, {0x23, 0x0E}, {0x44, 0x0E}, {0x67, 0x0E}, {0x8B, 0x0E}, 
        {0x59, 0x11}, {0x6D, 0x11}, {0x83, 0x11}, {0x9A, 0x11}, {0xB2, 0x11}, {0xCC, 0x11}, {0xE8, 0x11}, {0x05, 0x12}, {0x23, 0x12}, {0x44, 0x12}, {0x67, 0x12}, {0x8B, 0x12}, 
        {0x59, 0x15}, {0x6D, 0x15}, {0x83, 0x15}, {0x9A, 0x15}, {0xB2, 0x15}, {0xCC, 0x15}, {0xE8, 0x15}, {0x05, 0x16}, {0x23, 0x16}, {0x44, 0x16}, {0x67, 0x16}, {0x8B, 0x16},
        {0x59, 0x19}, {0x6D, 0x19}, {0x83, 0x19}, {0x9A, 0x19}, {0xB2, 0x19}, {0xCC, 0x19}, {0xE8, 0x19}, {0x05, 0x1A}, {0x23, 0x1A}, {0x44, 0x1A}, {0x67, 0x1A}, {0x8B, 0x1A},
        {0x59, 0x1D}, {0x6D, 0x1D}, {0x83, 0x1D}, {0x9A, 0x1D}
    };

static const uint8_t g_instrument_registers[NR_OF_CHANNELS][INSTRUMENT_DATA_SIZE] =
    {
        {0x20, 0x23, 0x40, 0x43, 0x60, 0x63, 0x80, 0x83, 0xA0, 0xB0, 0xC0},
        {0x21, 0x24, 0x41, 0x44, 0x61, 0x64, 0x81, 0x84, 0xA1, 0xB1, 0xC1},
        {0x22, 0x25, 0x42, 0x45, 0x62, 0x65, 0x82, 0x85, 0xA2, 0xB2, 0xC2},

        {0x28, 0x2B, 0x48, 0x4B, 0x68, 0x6B, 0x88, 0x8B, 0xA3, 0xB3, 0xC3},
        {0x29, 0x2C, 0x49, 0x4C, 0x69, 0x6C, 0x89, 0x8C, 0xA4, 0xB4, 0xC4},
        {0x2A, 0x2D, 0x4A, 0x4D, 0x6A, 0x6D, 0x8A, 0x8D, 0xA5, 0xB5, 0xC5},

        {0x30, 0x33, 0x50, 0x53, 0x70, 0x73, 0x90, 0x93, 0xA6, 0xB6, 0xC6},
        {0x31, 0x34, 0x51, 0x54, 0x71, 0x74, 0x91, 0x94, 0xA7, 0xB7, 0xC7},
        {0x32, 0x35, 0x52, 0x55, 0x72, 0x75, 0x92, 0x95, 0xA8, 0xB8, 0xC8}
    };

static const uint8_t g_preset_registers[NR_OF_CHANNELS][PRESET_DATA_SIZE] =
    {
        {0x20, 0x23, 0x40, 0x43, 0x60, 0x63, 0x80, 0x83, 0xC0, 0xE0, 0xE3},
        {0x21, 0x24, 0x41, 0x44, 0x61, 0x64, 0x81, 0x84, 0xC1, 0xE1, 0xE4},
        {0x22, 0x25, 0x42, 0x45, 0x62, 0x65, 0x82, 0x85, 0xC2, 0xE2, 0xE5},

        {0x28, 0x2B, 0x48, 0x4B, 0x68, 0x6B, 0x88, 0x8B, 0xC3, 0xE8, 0xEB},
        {0x29, 0x2C, 0x49, 0x4C, 0x69, 0x6C, 0x89, 0x8C, 0xC4, 0xE9, 0xEC},
        {0x2A, 0x2D, 0x4A, 0x4D, 0x6A, 0x6D, 0x8A, 0x8D, 0xC5, 0xEA, 0xED},

        {0x30, 0x33, 0x50, 0x53, 0x70, 0x73, 0x90, 0x93, 0xC6, 0xF0, 0xF3},
        {0x31, 0x34, 0x51, 0x54, 0x71, 0x74, 0x91, 0x94, 0xC7, 0xF1, 0xF4},
        {0x32, 0x35, 0x52, 0x55, 0x72, 0x75, 0x92, 0x95, 0xC8, 0xF2, 0xF5}
    };

static const uint8_t g_drum_presets[] =
    {
        0x30, 0x28, 0x00, 0x24, 0x22, 0x21, 0x38, 0x34, 0x32, 0x31, 0x2C, 0x2A, 0x29, 0x26, 0x25, 0x23
    };

static const uint8_t g_music_drum_patches[][INSTRUMENT_DATA_SIZE] =
    {
        {0x01, 0x01, 0x0E, 0x04, 0xDF, 0xF8, 0x6A, 0x6D},     /* BD      */
        {0x01, 0x01, 0x00, 0x00, 0xC8, 0xD8, 0xA7, 0x48},     /* SD / HH */
        {0x05, 0x01, 0x00, 0x00, 0xF8, 0xAA, 0x59, 0x55},     /* TM / TC */
    };

static const uint8_t g_music_voice_patches[MUSIC_NR_OF_PRESETS][MUSIC_INSTRUMENT_DATA_SIZE] =
    {
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},      /* User patch (irrelevant) */
        {0x71, 0x61, 0x1E, 0x17, 0xD0, 0x78, 0x00, 0x17},
        {0x13, 0x41, 0x1A, 0x0D, 0xD8, 0xF7, 0x23, 0x13},
        {0x13, 0x01, 0x99, 0x00, 0xF2, 0xC4, 0x11, 0x23},
        {0x31, 0x61, 0x0E, 0x07, 0xA8, 0x64, 0x70, 0x27},
        {0x32, 0x21, 0x1E, 0x06, 0xE0, 0x76, 0x00, 0x28},
        {0x31, 0x22, 0x16, 0x05, 0xE0, 0x71, 0x00, 0x18},
        {0x21, 0x61, 0x1D, 0x07, 0x82, 0x81, 0x10, 0x07},
        {0x23, 0x21, 0x2D, 0x14, 0xA2, 0x72, 0x00, 0x07},
        {0x61, 0x61, 0x1B, 0x06, 0x64, 0x65, 0x10, 0x17},
        {0x41, 0x61, 0x0B, 0x18, 0x85, 0xF7, 0x71, 0x07},
        {0x13, 0x01, 0x83, 0x11, 0xFA, 0xE4, 0x10, 0x04},
        {0x17, 0xC1, 0x24, 0x07, 0xF8, 0xF8, 0x22, 0x12},
        {0x61, 0x50, 0x0C, 0x05, 0xC2, 0xF5, 0x20, 0x42},
        {0x01, 0x01, 0x55, 0x03, 0xC9, 0x95, 0x03, 0x02},
        {0x61, 0x41, 0x89, 0x03, 0xF1, 0xE4, 0x40, 0x13}
    };

static const SAMPLE_BLOCK g_sample_blocks[NR_OF_SAMPLE_BLOCKS] = 
    {
        {0x0000, 0x1FFF}, {0x2000, 0x1FFF}, {0x4000, 0x1FFF}, {0x6000, 0x1FFF},
        {0x8000, 0x1FFF}, {0xA000, 0x1FFF}, {0xC000, 0x1FFF}, {0xE000, 0x1FFF},
        {0x0000, 0x3FFF}, {0x4000, 0x3FFF}, {0x8000, 0x3FFF}, {0xC000, 0x3FFF},
        {0x0000, 0x7FFF}, {0x8000, 0x7FFF},
        {0x0000, 0xFFFF}
    };

static const uint8_t g_pitch_table[][2] =
    {
         {0xB3, 0xA2}, {0xB4, 0x46}, {0xB4, 0xFB}, {0xB5, 0xC3}, {0xB6, 0x9E}, {0xB7, 0x78},
         {0xC0, 0x32}, {0xC0, 0xB1}, {0xC1, 0x30}, {0xC1, 0xC2}, {0xC2, 0x5C}, {0xC2, 0xF6},
         {0xC3, 0x9B}, {0xC4, 0x47}, {0xC5, 0x05}, {0xC5, 0xCD}, {0xC6, 0x9E}, {0xC7, 0x78},
         {0xD0, 0x37}, {0xD0, 0xB1}, {0xD1, 0x35}, {0xD1, 0xC2}, {0xD2, 0x57}, {0xD2, 0xF2},
         {0xD3, 0x9B}, {0xD4, 0x47}, {0xD5, 0x06}, {0xD5, 0xCD}, {0xD6, 0x9A}, {0xD7, 0x79},
         {0xE0, 0x37}, {0xE0, 0xB3}, {0xE1, 0x33}, {0xE1, 0xC2}, {0xE2, 0x58}, {0xE2, 0xF4},
         {0xE3, 0x9D}, {0xE4, 0x49}, {0xE5, 0x06}, {0xE5, 0xCB}, {0xE6, 0x9A}, {0xE7, 0x7B},
         {0xF0, 0x36}, {0xF0, 0xB3}, {0xF1, 0x34}, {0xF1, 0xC2}, {0xF2, 0x59}, {0xF2, 0xF3},
         {0xF3, 0x9C}, {0xF4, 0x48}, {0xF5, 0x06}, {0xF5, 0xCB}, {0xF6, 0x9B}, {0xF7, 0x7B},
         {0x00, 0x36}, {0x00, 0xB3}, {0x01, 0x34}, {0x01, 0xC2}, {0x02, 0x59}, {0x02, 0xF4}
    };

static MUS_HEADER *g_mus_header;

static char g_song_name[SONG_NAME_LENGTH + 1];
static char g_author[AUTHOR_NAME_LENGTH + 1];

static MUS_STATUS_TABLE g_mus_status_table[NR_OF_CHANNELS];

static uint8_t g_speed;
static uint8_t g_speed_count;

static uint8_t *g_pattern_data;
static uint8_t *g_song_end;

static bool g_play_samples;

static uint8_t g_mus_volume_table[NR_OF_CHANNELS];

static uint8_t g_unpack_segment;

static const uint8_t g_factors[] = {0x39, 0x39, 0x39, 0x39, 0x4D, 0x66, 0x80, 0x99};

void write_sample_headers();
bool load_sample_kit(const char* file_name, const char* extension);

void play_sample(uint8_t frequency, uint8_t volume, uint8_t sample_number);

void opll_instrument_to_opl_preset(uint8_t *opl_instrument_preset, const uint8_t *instrument_data, uint8_t volume, uint8_t pan_preset);
void opll_to_opl_fm_1(const uint8_t channel, const uint8_t *instrument_data);
void opll_to_opl_fm_2(const uint8_t channel, const uint8_t *instrument_data);

void frequency_up(uint8_t channel, uint8_t value);
void frequency_down(uint8_t channel, uint8_t value);
void handle_pitch_bend();

void note_on(uint8_t channel, uint8_t data);
void note_off(uint8_t channel);

void detune_up(uint8_t channel);
void detune_down(uint8_t channel);
void pitch_bend_up(uint8_t channel, uint8_t data);
void pitch_bend_down(uint8_t channel, uint8_t data);

void change_tempo(uint8_t data);
void change_volume(uint8_t channel, uint8_t data);
void change_brightness(uint8_t channel, uint8_t data);
