/*
 * RoboPlay for MSX
 * Copyright (C) 2020 by RoboSoft Inc.
 * 
 * roboplay.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <fusion-c/header/msx_fusion.h>
#include <fusion-c/header/vars_msxDos.h>
#include <fusion-c/header/rammapper.h>

#include "opl4.h"
#include "psg.h"
#include "scc.h"
#include "opm.h"
#include "memory.h"
#include "file.h"
#include "m3u.h"
#include "keyboard.h"
#include "player.h"
#include "roboplay.h"

static roboplay_play_mode g_play_mode = ROBOPLAY_PLAY_MODE_SINGLE_FILE;

static char *g_extension;
bool   g_use_wildcard;

/*
 * main
 */
int main(char **argv, int argc)
{
    srand(RealTimer());

    show_info();
    check_arguments(argv, argc);

    init_opl4_device();
    init_scc_device();
//    init_opm_device();

    psg_reset();

    if (g_play_mode == ROBOPLAY_PLAY_MODE_M3U || g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM)
    {
        printf("\n\rUsing M3U playlist: %s", g_m3u_name);
        if (g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM) printf(", Randomized");
        printf("\n\r");

        m3u_open_file(g_m3u_name);
        if(g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM)
            m3u_get_random_song(g_current_song_name, MAX_FILE_NAME_LENGTH);
        else
            m3u_get_next_song(g_current_song_name, MAX_FILE_NAME_LENGTH);

        find_extension(g_current_song_name);
        find_player_name();
    }

    printf("\nTotal free memory found: %dK\n\r", pre_allocate_ram_segments() * 16);

    while(play_song())
    {
        if(g_use_wildcard)
        {
            if(!find_next_file()) break;
        }
        else
        {
            if(g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM)
                m3u_get_random_song(g_current_song_name, MAX_FILE_NAME_LENGTH);
            else
            {
                if(!m3u_get_next_song(g_current_song_name, MAX_FILE_NAME_LENGTH)) break;
            }
        }
     
        find_extension(g_current_song_name);
        find_player_name();
    }

    if (g_play_mode == ROBOPLAY_PLAY_MODE_M3U || g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM)
    {
        m3u_close_file();
    }

    reset_used_ram_segments();
    KillKeyBuffer();

    return 0;
}

#ifdef __DEBUG
void debug_string(char *const str)
{
    printf("%s", str);
}

void debug_hex(const uint16_t num)
{
    PrintHex(num);
}
#endif

void show_info()
{
    printf("RoboPlay v1.3 - MSX Multi music format player\n\r");
    printf("Copyright (C) 2021 RoboSoft Inc.\n\r");
}

void show_song_info()
{
    printf("\n\r");
    printf("Title       : %s\n\r", g_roboplay_interface->get_title());
    printf("Author      : %s\n\r", g_roboplay_interface->get_author());
    printf("Description : %s\n\r", g_roboplay_interface->get_description());

    if(g_roboplay_interface->get_subsongs() > 1)
    {
        printf("Number of subsongs: %d\n\r", g_roboplay_interface->get_subsongs());
    }
        
    printf("\nNow playing ...ESC to stop");

    if(g_roboplay_interface->get_subsongs() > 1)
    {
        printf(", LEFT/RIGHT for subsong");
    }

     if (g_play_mode == ROBOPLAY_PLAY_MODE_M3U || g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM || g_use_wildcard)
     {
        printf(", SPACE for next song");
     }
    printf("\n\r");
}

void check_arguments(char **argv, int argc)
{
    if (!argc)
    {
        printf("\n\rUsage:\n\r  RoboPlay [<player>.PLY] <song>.EXT\n\r");
        printf("or\n\r  RoboPlay <name>.M3U [/R]\n\r");
        printf("\n\rPlayer name is optional, wildcards in the song name are allowed.");
        printf("\n\rWhen no player is provided, the file extension is used for identification\n\r");
        printf("\n\rUse the '/R' option to randomize the M3U playlist\n\r");
        Exit(0);
    }

    g_use_wildcard = false;
    for (uint8_t i = 0; i < strlen(argv[0]); i++)
    {
      argv[0][i] = toupper(argv[0][i]);
      if(argv[0][i] == '*' || argv[0][i] == '?') g_use_wildcard = true;
    } 
    if (argc > 1) for (uint8_t i = 0; i < strlen(argv[1]); i++) argv[1][i] = toupper(argv[1][i]);

    find_extension(argv[0]);

    if (!strcmp(g_extension, "M3U"))
    {
        /* Use M3U playlist */
        g_play_mode = ROBOPLAY_PLAY_MODE_M3U;

        strcpy(g_m3u_name, argv[0]);
        
        if (argc > 1)
        {
            if (!strcmp(argv[1], "/R")) g_play_mode = ROBOPLAY_PLAY_MODE_M3U_RANDOM;
        }
    }
    else
    {
        if (argc == 1)
        {
            if(!g_use_wildcard)
                strcpy(g_current_song_name, argv[0]);
            else
            {
                find_first_file(argv[0]);
                find_extension(g_current_song_name);
            }

            /* Determine player based on extension */
            find_player_name();
        }
        else
        {
            if(!g_use_wildcard)
                strcpy(g_current_song_name, argv[1]);
            else
                find_first_file(argv[1]);

            /* Player provided as argument*/
            get_program_path(g_current_player_name);
            strcat(g_current_player_name, argv[0]);
        }
    }
}

void find_player_name()
{
    /* Find player name based on extension of current song name */
    get_program_path(g_current_player_name);
    strcat(g_current_player_name, g_extension);
    strcat(g_current_player_name, ".PLY");
}

void find_extension(char *name)
{
    uint8_t found = strlen(name) - 1;
    while(name[found] != '.' && found > 0) found--;

    g_extension = &name[found + 1];
}

void init_opl4_device()
{
    printf("\n\rDetecting OPL4 device");

    if (!opl4_detect())
    {
        printf(" ...Not Found\n\r");
        Exit(0);
    }
    else
    {
        opl4_reset();
        printf(" ...Found, sample memory: %iK\n\r", opl4_sample_ram_banks() * 64);
    }

    opl4_reset();
}

void init_scc_device()
{
    printf("Detecting SCC  device");

    scc_find_slot();
    if(!scc_detected())
    {
        printf(" ...Not Found\n\r");
    }
    else
    {
        uint8_t scc_slot = scc_get_slot();
        if(scc_slot & 0x80)
            printf(" ...Found, located in slot %i-%i\n\r", scc_slot & 0x03, (scc_slot & 0x0C) >> 2);
        else
            printf(" ...Found, located in slot %i\n\r", scc_slot);
    }

    scc_reset();
}

void init_opm_device()
{
    printf("Detecting OPM  device");

    opm_find_slot();
    if(!opm_detected())
    {
        printf(" ...Not Found\n\r");
    }
    else
    {
        uint8_t opm_slot = opm_get_slot();
        if(opm_slot & 0x80)
            printf(" ...Found, located in slot %i-%i\n\r", opm_slot & 0x03, (opm_slot & 0x0C) >> 2);
        else
            printf(" ...Found, located in slot %i\n\r", opm_slot);
    }

    opm_reset();
}

void init_player()
{
    reset_used_ram_segments();

    load_player(g_current_player_name);

    printf("%s\n\r\n\r", g_roboplay_interface->get_player_info());

    g_roboplay_interface->open   = &file_open;
    g_roboplay_interface->read   = &file_read;
    g_roboplay_interface->close  = &file_close;
    g_roboplay_interface->exists = &file_exists;

    g_roboplay_interface->get_new_segment     = &get_free_ram_segment;
    g_roboplay_interface->set_segment         = &set_active_ram_segment;

    g_roboplay_interface->update_refresh = &set_refresh;

    g_roboplay_interface->opl_set_mode = &set_opl_mode;

    g_roboplay_interface->opl_write_fm_1 = &write_opl_fm_1;
    g_roboplay_interface->opl_write_fm_2 = &write_opl_fm_2;

    g_roboplay_interface->opl_write_wave = &opl4_write_wave_register;
    g_roboplay_interface->opl_write_wave_data = &opl4_write_wave_data;

    g_roboplay_interface->opl_read_status = &opl4_read_status_register;
    g_roboplay_interface->opl_read_fm_1 = &opl4_read_fm_register_array_1;
    g_roboplay_interface->opl_read_fm_2 = &opl4_read_fm_register_array_2;
    g_roboplay_interface->opl_read_wave_register = &opl4_read_wave_register;

    g_roboplay_interface->opl_wait_for_load = &opl4_wait_for_ld;

    g_roboplay_interface->psg_write = &psg_write;
    g_roboplay_interface->psg_read = &psg_read;

    g_roboplay_interface->scc_set_waveform = &scc_set_waveform;
    g_roboplay_interface->scc_set_frequency = &scc_set_frequency;
    g_roboplay_interface->scc_set_volume = &scc_set_volume;
    g_roboplay_interface->scc_write_register = &scc_write_register;

    g_roboplay_interface->opm_write = &write_opm_fm;
    g_roboplay_interface->opm_read_status = &opm_read_status_register;

#ifdef __DEBUG
    g_roboplay_interface->debug_string = &debug_string;
    g_roboplay_interface->debug_hex = &debug_hex;
#endif    

}

void set_refresh()
{
    float refresh = g_roboplay_interface->get_refresh();

#ifndef __DEBUG    
    if (refresh > 50.0)
    {
        opl4_write_fm_register_array_1(OPL4_TIMER1_COUNT, 255 - (uint8_t)((1000.0 / refresh) / 0.08));
        opl4_write_fm_register_array_1(OPL4_TIMER_CONTROL, 0x21);
        opl4_write_fm_register_array_1(OPL4_TIMER_CONTROL, 0x80);
    }
    else
    {
        opl4_write_fm_register_array_1(OPL4_TIMER2_COUNT, 256 - (uint8_t)((1000.0 / refresh) / 0.320));
        opl4_write_fm_register_array_1(OPL4_TIMER_CONTROL, 0x42);
        opl4_write_fm_register_array_1(OPL4_TIMER_CONTROL, 0x80);
    }
#else
    printf("\n\rRefresh: %i\n\r", (uint16_t)refresh);
#endif
}

bool play_song()
{
    bool result = false;

    init_player();
    if(!g_roboplay_interface->load(g_current_song_name))
    {       
        printf("\n\rError: Not a valid file for this player\n\r");
        if(g_play_mode == ROBOPLAY_PLAY_MODE_SINGLE_FILE)
            Exit(__INTER);
        else return true;
    }

    show_song_info();

#ifndef __DEBUG
    __asm
    di
    __endasm;
#endif

    uint8_t subsong = 0;
    g_roboplay_interface->rewind(subsong);

    set_refresh();
    while(!is_key_pressed_ESC())
    {
        if((g_play_mode == ROBOPLAY_PLAY_MODE_M3U || g_play_mode == ROBOPLAY_PLAY_MODE_M3U_RANDOM) && is_key_pressed_SPACE())
        {
            result = true;
            break;
        }

        if(g_use_wildcard && is_key_pressed_SPACE())
        {
            result = true;
            break;
        }

        for(uint8_t i = 1; i < 7; i++)
        {
            if(is_number_key_pressed(i)) g_roboplay_interface->command(i);
        }

        if(!g_roboplay_interface->update())
            g_roboplay_interface->rewind(subsong);

        if(g_roboplay_interface->get_subsongs() > 1)
        {
            if(is_key_pressed_RIGHT() && subsong < g_roboplay_interface->get_subsongs())
            {
                subsong++;
                g_roboplay_interface->rewind(subsong);
            }

            if(is_key_pressed_LEFT() && subsong > 0)
            {
                subsong--;
                g_roboplay_interface->rewind(subsong);
            }
        }

#ifndef __DEBUG
        while(!(opl4_read_status_register() & 0x80));
        opl4_write_fm_register_array_1(0x04, 0x80);
#else
        while(!is_key_pressed_RIGHT());
#endif        
    }

    opl4_reset();
    scc_reset();
    psg_reset();
 
#ifndef __DEBUG
    __asm
    ei
    __endasm;
#endif

    return result;
}

void set_opl_mode(const roboplay_opl_mode mode)
{
    switch(mode)
    {
        case ROBOPLAY_OPL_MODE_OPL2:
            opl4_write_fm_register_array_2(0x05, 0x00);
            break;
        case ROBOPLAY_OPL_MODE_OPL3:
            opl4_write_fm_register_array_2(0x05, 0x01);
            break;
        case ROBOPLAY_OPL_MODE_OPL4:
            opl4_write_fm_register_array_2(0x05, 0x03);
            break;
    }
}

void write_opl_fm_1(const uint8_t reg, const uint8_t value)
{
#ifdef __DEBUG
    printf("1:%x:%x ", reg, value);
#endif

    if(reg == 2 || reg == 3 || reg == 4) return;

    opl4_write_fm_register_array_1(reg, value);
}

void write_opl_fm_2(const uint8_t reg, const uint8_t value)
{
#ifdef __DEBUG
    printf("2:%x:%x ", reg, value);
#endif

    opl4_write_fm_register_array_2(reg, value);
}

void write_opm_fm(const uint8_t reg, const uint8_t value)
{
#ifdef __DEBUG
    printf("%x:%x ", reg, value);
#endif

    if(reg == 0x11 || reg == 0x12 || reg == 0x13 || reg == 0x14) return;

    opm_write_register(reg, value);
}