// ____________________________
// ██▀▀█▀▀██▀▀▀▀▀▀▀█▀▀█        │  ▄▄▄   ▄▄   ▄ ▄ ▄▄▄▄ ▄▄▄   ▄▄ 
// ██  ▀  █▄  ▀██▄ ▀ ▄█ ▄▀▀ █  │  ██▄▀ ██ ▀ ██▀█ ██▄  ██▀█ ██ ▀
// █  █ █  ▀▀  ▄█  █  █ ▀▄█ █▄ │  ██   ▀█▄▀ ██ █ ██▄▄ ██ █ ▀█▄▀
// ▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀────────┘
//  by Guillaume 'Aoineko' Blanchard under CC BY-SA license
//─────────────────────────────────────────────────────────────────────────────
// PCM-Encoder replayer
//
// Copyright (C) 2006 Arturo Ragozini and Daniel Vik
//
//  This software is provided 'as-is', without any express or implied
//  warranty.  In no event will the authors be held liable for any damages
//  arising from the use of this software.
//
//  Permission is granted to anyone to use this software for any purpose,
//  including commercial applications, and to alter it and redistribute it
//  freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you must not
//     claim that you wrote the original software. If you use this software
//     in a product, an acknowledgment in the product documentation would be
//     appreciated but is not required.
//  2. Altered source versions must be plainly marked as such, and must not be
//     misrepresented as being the original software.
//  3. This notice may not be removed or altered from any source distribution.
//─────────────────────────────────────────────────────────────────────────────
#include "pcmenc.h"

#if (PCMENC_FREQ != PCMENC_NONE)
//-----------------------------------------------------------------------------
// Reset PSG registers (must not change HL!)
void PCM_Reset()
{
	__asm
        xor     a
        ld      bc, #0xFFA1
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        inc     a
        out     (c), b
        out     (P_PSG_REGS), a
        ld      b, #0xBF
        out     (c), b
	__endasm;		
}
#endif

#if (PCMENC_FREQ & PCMENC_8K)
//=============================================================================
// 8K REPLAYER
//=============================================================================
// Replayer core to play RLE encoded 8.000kHz samples generated by pcmenc
// pcmenc should use the following command line arguments:
//    pcmenc -dt1 31 -dt2 27 -dt3 388 file.wav
// and optionally -r to split sample into 8kB blocks for rom replayer

//-----------------------------------------------------------------------------
// Play a pcmenc sound (synchronously)
//-----------------------------------------------------------------------------
void PCM_Play_8K(u16 addr)
{
	addr; // HL
	
	PCM_Reset(); // don't change HL

__asm
		di
		ld		e, (hl)
		inc		hl
		ld		d, (hl)
		inc		hl
//-------------------------------------
// Plays one sample
// IN   HL - Encoded sample start address
//      DE - Sample length (#pcm samples)
//-------------------------------------
	PCM_8K_Play:
        ld      c, #PSG_PORT_WRITE
        push    hl
        ex      de, hl
        inc     h
        exx
        pop     hl
        ld      bc, #0x0010
        ld      de, #0x0000
        exx
        
	PCM_8K_Loop:
        exx                         // 5     -> 5

		// Calculate channel A volume
        ld      a, b                // 5
        sub     c                   // 5
        jr      nc, PCM_8K_WaitA    // 8/13
        ld      a, (hl)             // 8
        ld      b, a                // 5
        and     #0x0F               // 8
        inc     hl                  // 7
        exx                         // 5
        ld      b, a                // 5
        exx                         // 5     -> 61
	PCM_8K_DoneA:
        
		// Calculate channel B volume
        ld      a, d                // 5
        sub     c                   // 5
        jr      nc, PCM_8K_WaitB    // 8/13
        ld      a, (hl)             // 8
        ld      d, a                // 5
        and     #0x0F               // 8
        inc     hl                  // 7
        exx                         // 5
        ld      d, a                // 5
        exx                         // 5     -> 61
	PCM_8K_DoneB:
        
		// Calculate channel C volume
        ld      a, e                // 5
        sub     c                   // 5
        jr      nc, PCM_8K_WaitC    // 8/13
        ld      a, (hl)             // 8
        ld      e, a                // 5        
        and     #0x0F               // 8
        inc     hl                  // 7
        exx                         // 5
        ld      e, a                // 5     -> 56
	PCM_8K_DoneC:

        // Output channels
        ld      a, #8               // 8
        out     (PSG_PORT_REG), a   // 12
        inc     a                   // 5
        out     (c), b              // 15
        out     (PSG_PORT_REG), a   // 12
        inc     a                   // 5
        out     (c), d              // 15
        out     (PSG_PORT_REG), a   // 12
        out     (c), e              // 15   -> 99
        
        // Wait 147 t-cycles
        call    Wait147             // 147
        
        // Decrement length and return if zero
        dec     l                   // 5
        jr      nz, PCM_8K_Loop     // 8/13   -> 18   Total: 447
        dec     h                   // 5
        jp      nz, PCM_8K_Loop     // 11
		ei
        ret
        
PCM_8K_WaitA:
        ld      b, a                // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_8K_DoneA        // 13   -> 51 including branch
        
PCM_8K_WaitB:
        ld      d, a                // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_8K_DoneB        // 13   -> 51 including branch
								    
PCM_8K_WaitC:
        ld      e, a                // 5
        exx                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_8K_DoneC        // 11   -> 51 including branch


Wait147:
        inc     hl                  // 7
        dec     hl                  // 7
        inc     hl                  // 7
        dec     hl                  // 7
        inc     hl                  // 7
        dec     hl                  // 7
        inc     hl                  // 7
        dec     hl                  // 7
        inc     hl                  // 7
        dec     hl                  // 7
        inc     hl                  // 7
        dec     hl                  // 7
        inc     hl                  // 7
        dec     hl                  // 7
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        ret                         // 11    -> 147 including branch
	__endasm;
}        
#endif // (PCMENC_FREQ & PCMENC_8K)


#if (PCMENC_FREQ & PCMENC_11K)
//=============================================================================
// 11K REPLAYER
//=============================================================================
// Replayer core to play RLE encoded 11.025kHz samples generated by pcmenc
// pcmenc should use the following command line arguments:
//    pcmenc -dt1 32 -dt2 27 -dt3 265 file.wav
// and optionally -r to split sample into 8kB blocks for rom replayer

//-----------------------------------------------------------------------------
/// Play a pcmenc sound (synchronously)
//-----------------------------------------------------------------------------
void PCM_Play_11K(u16 addr)
{
	addr; // HL

	PCM_Reset(); // don't change HL

	__asm
		di
		ld		e, (hl)
		inc		hl
		ld		d, (hl)
		inc		hl
//-------------------------------------
// Plays one sample
// IN   HL - Encoded sample start address
//      DE - Sample length (#pcm samples)
//-------------------------------------
	PCM_11K_Play:
        ld      c, #PSG_PORT_WRITE
        push    hl
        ex      de, hl
        inc     h
        exx
        pop     hl
        ld      bc, #0x0010
        ld      de, #0x0000
        exx

	PCM_11K_Loop:
        exx                         // 5     -> 5

		// Calculate channel A volume
        ld      a, b                // 5
        sub     c                   // 5
        jr      nc, PCM_11K_WaitA   // 8/13
        ld      a, (hl)             // 8
        ld      b, a                // 5
        and     #0x0F               // 8
        inc     hl                  // 7
        exx                         // 5
        ld      b, a                // 5
        exx                         // 5     -> 61
	PCM_11K_DoneA:

		// Calculate channel B volume
        ld      a, d                // 5
        sub     c                   // 5
        jr      nc, PCM_11K_WaitB   // 8/13
        ld      a, (hl)             // 8
        ld      d, a                // 5
        and     #0x0F               // 8
        inc     hl                  // 7
        exx                         // 5
        ld      d, a                // 5
        exx                         // 5     -> 61
	PCM_11K_DoneB:

		// Calculate channel C volume
        ld      a, e                // 5
        sub     c                   // 5
        jr      nc, PCM_11K_WaitC   // 8/13
        ld      a, (hl)             // 8
        ld      e, a                // 5        
        and     #0x0F               // 8
        inc     hl                  // 7
        exx                         // 5
        ld      e, a                // 5     -> 56
	PCM_11K_DoneC:

        // Output channels
        ld      a, #8               // 8
        out     (PSG_PORT_REG), a   // 12
        inc     a                   // 5
        out     (c), b              // 15
        out     (PSG_PORT_REG), a   // 12
        inc     a                   // 5
        out     (c), d              // 15
        out     (PSG_PORT_REG), a   // 12
        out     (c), e              // 15   -> 99

        // Wait 25 t-cycles
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5

        // Decrement length and return if zero
        dec     l                   // 5
        jr      nz, PCM_11K_Loop    // 8/13   -> 18   Total: 325
        dec     h                   // 5
        jp      nz, PCM_11K_Loop    // 11
		ei
        ret

	PCM_11K_WaitA:
        ld      b, a                // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_11K_DoneA       // 13   -> 51 including branch

	PCM_11K_WaitB:
        ld      d, a                // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_11K_DoneB       // 13   -> 51 including branch

	PCM_11K_WaitC:
        ld      e, a                // 5
        exx                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_11K_DoneC       // 11   -> 51 including branch
	__endasm;
}
#endif // (PCMENC_FREQ & PCMENC_11K)


#if (PCMENC_FREQ & PCMENC_22K)
//=============================================================================
// 22K REPLAYER
//=============================================================================
// Replayer core to play RLE encoded 22.05kHz samples generated by pcmenc
// pcmenc should use the following command line arguments:
//    pcmenc -rto 2 -dt1 156 -dt2 27 -dt3 141 file.wav
// and optionally -r to split sample into 8kB blocks for rom replayer

//-----------------------------------------------------------------------------
/// Play a pcmenc sound (synchronously)
//-----------------------------------------------------------------------------
void PCM_Play_22K(u16 addr)
{
	addr; // HL

	PCM_Reset(); // don't change HL

	__asm
		di
		ld		e, (hl)
		inc		hl
		ld		d, (hl)
		inc		hl
//-------------------------------------
// Plays one sample
// IN   HL - Encoded sample start address
//      DE - Sample length (#pcm samples)
//-------------------------------------
	PCM_22K_Play:
        ld      c, #PSG_PORT_WRITE
        push    hl
        ex      de, hl
        inc     h
        exx
        pop     hl
        ld      bc, #0x0010
        ld      de, #0x0000
        exx
        
	PCM_22K_Loop:
        exx                         // 5     -> 5
        
		// Calculate and output channel A volume
        ld      a, b                // 5
        sub     #0x10               // 8
        jr      nc, PCM_22K_WaitA   // 8/13
        ld      a, #8               // 8
        out     (PSG_PORT_REG), a   // 12
        ld      a, (hl)             // 8
        inc     hl                  // 7
        ld      b, a                // 5
        and     #0x0F               // 8
        out     (PSG_PORT_WRITE), a // 12    -> 81  Out at 86 after loop start
	PCM_22K_DoneA:

		// Calculate channel B volume
        ld      a, d                // 5
        sub     c                   // 5
        jr      nc, PCM_22K_WaitB   // 8/13
        ld      a, (hl)             // 8
        inc     hl                  // 7
        ld      d, a                // 5
        and     #0x0F               // 8
        exx                         // 5
        ld      d, a                // 5
        exx                         // 5     -> 61
	PCM_22K_DoneB:
        
		// Calculate channel C volume
        ld      a, e                // 5
        sub     c                   // 5
        jr      nc, PCM_22K_WaitC   // 8/13
        ld      a, (hl)             // 8
        inc     hl                  // 7
        ld      e, a                // 5
        and     #0x0F               // 8
        exx                         // 5
        ld      e, a                // 5     -> 56
	PCM_22K_DoneC:

        // Output channels B and C
        ld      a, #9               // 8
        out     (PSG_PORT_REG), a   // 12
        inc     a                   // 5
        out     (c), d              // 15          Out at 157+86 after loop start
        out     (PSG_PORT_REG), a   // 12
        out     (c), e              // 15   -> 67  Out at 27+157+86 after loop start
        
        // Wait32
        or      #0                  // 8
        or      #0                  // 8
        or      #0                  // 8
        or      #0                  // 8
        
        // Decrement length and return if zero
        dec     l                   // 5
        jr      nz, PCM_22K_Loop    // 8/13   -> 18   Total: 325
        dec     h                   // 5
        jp      nz, PCM_22K_Loop    // 11
		ei
        ret
        
	PCM_22K_WaitA:
        ld      b, a                // 5
        inc     hl                  // 7
        dec     hl                  // 7
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_22K_DoneA       // 13   -> 60 including branch
        
	PCM_22K_WaitB:
        ld      d, a                // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_22K_DoneB       // 13   -> 51 including branch
        
	PCM_22K_WaitC:
        ld      e, a                // 5
        exx                         // 5
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_22K_DoneC       // 11   -> 51 including branch
	__endasm;
}
#endif // (PCMENC_FREQ & PCMENC_22K)


#if (PCMENC_FREQ & PCMENC_44K)
//=============================================================================
// 44K REPLAYER
//=============================================================================
// Replayer core to play RLE encoded 44.1kHz samples generated by pcmenc
// pcmenc should use the following command line arguments:
//    pcmenc -p 1 -rto 3 -dt1 73 -dt2 84 -dt3 87 file.wav
// and optionally -r to split sample into 8kB blocks for rom replayer

//-----------------------------------------------------------------------------
/// Play a pcmenc sound (synchronously)
//-----------------------------------------------------------------------------
void PCM_Play_44K(u16 addr)
{
	addr; // HL

	PCM_Reset(); // don't change HL

	__asm
		di
		ld		e, (hl)
		inc		hl
		ld		d, (hl)
		inc		hl
//-------------------------------------
// Plays one sample
// IN   HL - Encoded sample start address
//      DE - Sample length (#pcm samples)
//-------------------------------------
	PCM_44K_Play:
        push    hl
        ld      h, d
        inc     h
        ld      a, e
        exx
        pop     hl
        ld      b, a
        ld      c, #0x00
        ld      de, #0x0000

	PCM_44K_Loop:
		// Calculate channel A volume
        ld      a, c                // 5
        sub     #0x20               // 8
        jr      nc, PCM_44K_WaitA   // 8/13
        ld      a, #8               // 8
        out     (PSG_PORT_REG), a   // 12
        ld      a, (hl)             // 8
        inc     hl                  // 7
        ld      c, a                // 5
        out     (PSG_PORT_WRITE), a // 12    -> 73
	PCM_44K_DoneA:

		// Calculate channel B volume
        ld      a, d                // 5
        sub     #0x20               // 8
        jr      nc, PCM_44K_WaitB   // 8/13
        ld      a, #9               // 8
        out     (PSG_PORT_REG), a   // 12
        ld      a, (hl)             // 8
        inc     hl                  // 7
        ld      d, a                // 5
        out     (PSG_PORT_WRITE), a // 12    -> 73
	PCM_44K_DoneB:

        // Delay 11 t-cycles
        jp      PCM_44K_Dummy       // 11    -> 11
	PCM_44K_Dummy:

		// Calculate channel C volume
        ld      a, e                // 5
        sub     #0x20               // 8
        jr      nc, PCM_44K_WaitC   // 8/13
        ld      a, #10              // 8
        out     (PSG_PORT_REG), a   // 12
        ld      a, (hl)             // 8
        inc     hl                  // 7
        ld      e, a                // 5
        out     (PSG_PORT_WRITE), a // 12    -> 73
	PCM_44K_DoneC:
	
        // Decrement length and return if zero
        djnz    PCM_44K_Loop        // 9/14  -> 14   Total: 244 (244)
        exx                         // 5
        dec     h                   // 5
        exx                         // 5     -> 5
        jp      nz, PCM_44K_Loop    // 11
		ei
        ret
        
	PCM_44K_WaitA:
        ld      c, a                // 5
        inc     hl                  // 7
        dec     hl                  // 7
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_44K_DoneA       // 13   -> 60 including branch
        
	PCM_44K_WaitB:
        ld      d, a                // 5
        inc     hl                  // 7
        dec     hl                  // 7
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_44K_DoneB       // 13   -> 60 including branch
        
	PCM_44K_WaitC:
        ld      e, a                // 5
        inc     hl                  // 7
        dec     hl                  // 7
        nop                         // 5
        nop                         // 5
        nop                         // 5
        jr      PCM_44K_DoneC       // 11   -> 60 including branch
	__endasm;
}
#endif // (PCMENC_FREQ & PCMENC_44K)


