unit msxMEM;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

var
  //메모리(0000h~FFFFh)
  MSX_MEM: Array[0..4-1, 0..65536-1] Of Byte; //0~255 [0~3, 0~65535]

  //Memory Page
  Gi_MEM_PAGE0: ShortInt; //-128~127
  Gi_MEM_PAGE1: ShortInt;
  Gi_MEM_PAGE2: ShortInt;
  Gi_MEM_PAGE3: ShortInt;

  //MSX2 - 확장 Page
  Gi_MEM_EXT_PAGE0: ShortInt;
  Gi_MEM_EXT_PAGE1: ShortInt;
  Gi_MEM_EXT_PAGE2: ShortInt;
  Gi_MEM_EXT_PAGE3: ShortInt;

  //Mega ROM Slot#1
  MSX_ROM: Array[0..4194304-1] Of Byte; //0~255 [0~4194303(4MB)]
  Gi_Big32K_ROM: ShortInt;

  //MSX2 - Mapper Memory RAM Slot#2
  MSX_RAM: Array[0..4194304-1] Of Byte; //0~255 [0~4194303(4MB)]
  Gi_RAM_PAGE0: Byte;
  Gi_RAM_PAGE1: Byte;
  Gi_RAM_PAGE2: Byte;
  Gi_RAM_PAGE3: Byte;

  procedure MSX_INIT_MEM;
  function GET_MSX_MEMORY_SLOT2(oAddr: Word): Byte;
  procedure PUT_MSX_MEMORY_SLOT2(oAddr: Word; iVal: Byte);
  function GET_MSX_MEMORY(oAddr: Word): Byte;
  procedure PUT_MSX_MEMORY(oAddr: Word; iVal: Byte);
  function GET_MSX_MEMORY_16(oAddr: Word): Word;
  procedure PUT_MSX_MEMORY_16(oAddr: Word; oVal: Word);
  procedure MSX_MEMORY_CHANGE(iPage: ShortInt; iSeg: Byte);
  function GET_MSX_MEMORY_MAPPER(oAddr: Word; iPage: ShortInt): Byte;
  procedure PUT_MSX_MEMORY_MAPPER(oAddr: Word; iVal: Byte; iPage: ShortInt);
  procedure Move_Rom_To_Page_Konami4(iBank: ShortInt; iPage: Byte);
  procedure Move_Rom_To_Page_Konami5(iBank: ShortInt; iPage: Byte);
  procedure Move_Rom_To_Page_Ascii8(iBank: ShortInt; iPage: Byte);
  procedure Move_Rom_To_Page_Ascii16(iBank: ShortInt; iPage: Byte);
  procedure Move_Rom_To_Page_80_in_1(iBank: ShortInt; iPage: Byte);
  procedure Move_Rom_To_Page_126_in_1(iBank: ShortInt; iPage: Byte);
  function foGetBankAddress8(iBank: ShortInt): LongInt;
  function foGetPageAddress8(iPage: Byte): LongInt;
  function foGetPageAddress16(iPage: Byte): LongInt;

implementation

uses
  msxMSX, msxTBL;

procedure MSX_INIT_MEM;
var
  i, j: Integer;
begin
  for i := 0 to 3 do
    for j := 0 to 65535 do
      MSX_MEM[i, j] := 0;

  Gi_MEM_PAGE0 := 0;
  Gi_MEM_PAGE1 := 0;
  Gi_MEM_PAGE2 := 0;
  Gi_MEM_PAGE3 := 0;

  Gi_MEM_EXT_PAGE0 := 0;
  Gi_MEM_EXT_PAGE1 := 0;
  Gi_MEM_EXT_PAGE2 := 0;
  Gi_MEM_EXT_PAGE3 := 0;

  for i := 0 to 4194303 do  MSX_ROM[i] := 0;

  Gi_Big32K_ROM := 0;

  for i := 0 to 4194303 do  MSX_RAM[i] := 0;

  Gi_RAM_PAGE0 := 3;
  Gi_RAM_PAGE1 := 2;
  Gi_RAM_PAGE2 := 1;
  Gi_RAM_PAGE3 := 0;
end;

function GET_MSX_MEMORY_SLOT2(oAddr: Word): Byte;
begin
  if (GiMSX2 = 0) then //MSX1
    Result := MSX_MEM[2, oAddr]
  else //MSX2
    Result := GET_MSX_MEMORY_MAPPER(oAddr, -1)
  ;
end;

procedure PUT_MSX_MEMORY_SLOT2(oAddr: Word; iVal: Byte);
begin
  if (GiMSX2 = 0) then //MSX1
    MSX_MEM[2, oAddr] := iVal
  else //MSX2
    PUT_MSX_MEMORY_MAPPER(oAddr, iVal, -1)
  ;
end;

function GET_MSX_MEMORY(oAddr: Word): Byte;
var
  iRet: Byte;
begin
  iRet := 0;

  if ((oAddr >= 0) And (oAddr <= 16383)) then //Page#0 [0000~3FFF]
    begin
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE0 = 2)) then //MSX2 Mapper RAM
        iRet := GET_MSX_MEMORY_MAPPER(oAddr, 0)
      else //RAM
        iRet := MSX_MEM[Gi_MEM_PAGE0, oAddr]
      ;
    end;

  if ((oAddr >= 16384) And (oAddr <= 32767)) then //Page#1 [4000~7FFF]
    begin
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE1 = 2)) then //MSX2 Mapper RAM
        iRet := GET_MSX_MEMORY_MAPPER(oAddr, 1)
      else //RAM
        iRet := MSX_MEM[Gi_MEM_PAGE1, oAddr]
      ;
    end;

  if ((oAddr >= 32768) And (oAddr <= 49151)) then //Page#2 [8000~BFFF]
    begin
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE2 = 2)) then //MSX2 Mapper RAM
        iRet := GET_MSX_MEMORY_MAPPER(oAddr, 2)
      else //RAM
        iRet := MSX_MEM[Gi_MEM_PAGE2, oAddr]
      ;
    end;

  if ((oAddr >= 49152) And (oAddr <= 65535)) then //Page#3 [C000~FFFF]
    begin
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE3 = 2)) then //MSX2 Mapper RAM
        iRet := GET_MSX_MEMORY_MAPPER(oAddr, 3)
      else //RAM
        iRet := MSX_MEM[Gi_MEM_PAGE3, oAddr]
      ;
    end;

  Result := iRet;
end;

procedure PUT_MSX_MEMORY(oAddr: Word; iVal: Byte);
label
  ExitProc;
begin
  if (Gi_Big32K_ROM = 1) then //Mega Rom Mapper
    begin
      if (Gi_ROM_TYPE = 0) then
        begin
          //Konami without SCC (a.k.a.Konami4) - [8KB]
          if (Gi_MEM_PAGE1 = 1) then
            begin
              if ((oAddr >= $4000) And (oAddr <= $5FFF)) then Move_Rom_To_Page_Konami4(1, iVal);
              if ((oAddr >= $6000) And (oAddr <= $7FFF)) then Move_Rom_To_Page_Konami4(2, iVal);
            end;
          if (Gi_MEM_PAGE2 = 1) then
            begin
              if ((oAddr >= $8000) And (oAddr <= $9FFF)) then Move_Rom_To_Page_Konami4(3, iVal);
              if ((oAddr >= $A000) And (oAddr <= $BFFF)) then Move_Rom_To_Page_Konami4(4, iVal);
            end;
      end;
      if (Gi_ROM_TYPE = 1) then
        begin
          //Konami with SCC (a.k.a.Konami5) - [8KB]
          if (Gi_MEM_PAGE1 = 1) then
            begin
              if ((oAddr >= $5000) And (oAddr <= $57FF)) then Move_Rom_To_Page_Konami5(1, iVal);
              if ((oAddr >= $7000) And (oAddr <= $77FF)) then Move_Rom_To_Page_Konami5(2, iVal);
            end;
          if (Gi_MEM_PAGE2 = 1) then
            begin
              if ((oAddr >= $9000) And (oAddr <= $97FF)) then Move_Rom_To_Page_Konami5(3, iVal);
              if ((oAddr >= $B000) And (oAddr <= $B7FF)) then Move_Rom_To_Page_Konami5(4, iVal);
            end;
        end;
      if (Gi_ROM_TYPE = 2) then
        begin
          //ASCII 8Kb - [8KB]
          if (Gi_MEM_PAGE1 = 1) then
            begin
              if ((oAddr >= $6000) And (oAddr <= $67FF)) then Move_Rom_To_Page_Ascii8(1, iVal);
              if ((oAddr >= $6800) And (oAddr <= $6FFF)) then Move_Rom_To_Page_Ascii8(2, iVal);
              if ((oAddr >= $7000) And (oAddr <= $77FF)) then Move_Rom_To_Page_Ascii8(3, iVal);
              if ((oAddr >= $7800) And (oAddr <= $7FFF)) then Move_Rom_To_Page_Ascii8(4, iVal);
            end;
        end;
      if (Gi_ROM_TYPE = 3) then
        begin
          //ASCII 16Kb - [16KB]
          if (Gi_MEM_PAGE1 = 1) then
            begin
              if ((oAddr >= $6000) And (oAddr <= $67FF)) then Move_Rom_To_Page_Ascii16(1, iVal);
              if ((oAddr >= $7000) And (oAddr <= $77FF)) then Move_Rom_To_Page_Ascii16(3, iVal);
            end;
        end;
      if (Gi_ROM_TYPE = 4) then
        begin
          //64-in-1 And 80-in-1 - [8KB]
          if (Gi_MEM_PAGE1 = 1) then
            begin
              if (oAddr = $4000) then Move_Rom_To_Page_80_in_1(1, iVal);
              if (oAddr = $4001) then Move_Rom_To_Page_80_in_1(2, iVal);
              if (oAddr = $4002) then Move_Rom_To_Page_80_in_1(3, iVal);
              if (oAddr = $4003) then Move_Rom_To_Page_80_in_1(4, iVal);
            end;
        end;
      if (Gi_ROM_TYPE = 5) then
        begin
          //126-in-1 - [16KB]
          if (Gi_MEM_PAGE1 = 1) then
            begin
              if (oAddr = $4000) then Move_Rom_To_Page_126_in_1(1, iVal);
              if (oAddr = $4001) then Move_Rom_To_Page_126_in_1(3, iVal);
            end;
        end;
    end;

  if ((oAddr >= 0) And (oAddr <= 16383)) then //Page#0 [0000~3FFF]
    begin
      if (Gi_MEM_PAGE0 = 0) then goto ExitProc; //Slot#0 ROM
      if (Gi_MEM_PAGE0 = 1) then goto ExitProc; //Slot#1 ROM
      if (Gi_MEM_PAGE0 = 3) then goto ExitProc; //Slot#3 ROM
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE0 = 2)) then //MSX2 Mapper RAM
        begin
          PUT_MSX_MEMORY_MAPPER(oAddr, iVal, 0);
        end
      else //RAM
        begin
          if (Gi_MEM_PAGE0 = 2) then MSX_MEM[Gi_MEM_PAGE0, oAddr] := iVal;
        end;
    end;

  if ((oAddr >= 16384) And (oAddr <= 32767)) then //Page#1 [4000~7FFF]
    begin
      if (Gi_MEM_PAGE1 = 0) then goto ExitProc; //Slot#0 ROM
      if (Gi_MEM_PAGE1 = 1) then goto ExitProc; //Slot#1 ROM
      if (Gi_MEM_PAGE1 = 3) then goto ExitProc; //Slot#3 ROM
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE1 = 2)) then //MSX2 Mapper RAM
        begin
          PUT_MSX_MEMORY_MAPPER(oAddr, iVal, 1);
        end
      else //RAM
        begin
          if (Gi_MEM_PAGE1 = 2) then MSX_MEM[Gi_MEM_PAGE1, oAddr] := iVal;
        end;
    end;

  if ((oAddr >= 32768) And (oAddr <= 49151)) then //Page#2 [8000~BFFF]
    begin
      if (Gi_MEM_PAGE2 = 0) then goto ExitProc; //Slot#0 ROM
      if (Gi_MEM_PAGE2 = 1) then goto ExitProc; //Slot#1 ROM
      if (Gi_MEM_PAGE2 = 3) then goto ExitProc; //Slot#3 ROM
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE2 = 2)) then //MSX2 Mapper RAM
        begin
          PUT_MSX_MEMORY_MAPPER(oAddr, iVal, 2);
        end
      else //RAM
        begin
          if (Gi_MEM_PAGE2 = 2) then MSX_MEM[Gi_MEM_PAGE2, oAddr] := iVal;
        end;
    end;

  if ((oAddr >= 49152) And (oAddr <= 65535)) then //Page#3 [C000~FFFF]
    begin
      if (Gi_MEM_PAGE3 = 0) then goto ExitProc; //Slot#0 ROM
      if (Gi_MEM_PAGE3 = 1) then goto ExitProc; //Slot#1 ROM
      if (Gi_MEM_PAGE3 = 3) then goto ExitProc; //Slot#3 ROM
      if ((GiMSX2 = 1) And (Gi_MEM_PAGE3 = 2)) then //MSX2 Mapper RAM
        begin
          PUT_MSX_MEMORY_MAPPER(oAddr, iVal, 3);
        end
      else //RAM
        begin
          if (Gi_MEM_PAGE3 = 2) then MSX_MEM[Gi_MEM_PAGE3, oAddr] := iVal;
        end;
    end;

ExitProc:
end;

function GET_MSX_MEMORY_16(oAddr: Word): Word;
var
  h, l: Byte;
begin
  l := GET_MSX_MEMORY(oAddr); oAddr := ((oAddr + 1) And $FFFF);
  h := GET_MSX_MEMORY(oAddr);

  Result := ((h shl 8) Or l);
end;

procedure PUT_MSX_MEMORY_16(oAddr: Word; oVal: Word);
var
  h, l: Byte;
begin
  h := (oVal shr 8);
  l := (oVal And $00FF);

  PUT_MSX_MEMORY(oAddr, l); oAddr := ((oAddr + 1) And $FFFF);
  PUT_MSX_MEMORY(oAddr, h);
end;

procedure MSX_MEMORY_CHANGE(iPage: ShortInt; iSeg: Byte);
begin
  if (iPage = 0) then Gi_RAM_PAGE0 := iSeg;
  if (iPage = 1) then Gi_RAM_PAGE1 := iSeg;
  if (iPage = 2) then Gi_RAM_PAGE2 := iSeg;
  if (iPage = 3) then Gi_RAM_PAGE3 := iSeg;
end;

function GET_MSX_MEMORY_MAPPER(oAddr: Word; iPage: ShortInt): Byte;
var
  iMapPage: Byte;
begin
  iMapPage := 0;

  if (iPage = (-1)) then
    begin
      if ((oAddr >= 0) And (oAddr <= 16383)) then iPage := 0; //Page#0 [0000~3FFF]
      if ((oAddr >= 16384) And (oAddr <= 32767)) then iPage := 1; //Page#1 [4000~7FFF]
      if ((oAddr >= 32768) And (oAddr <= 49151)) then iPage := 2; //Page#2 [8000~BFFF]
      if ((oAddr >= 49152) And (oAddr <= 65535)) then iPage := 3; //Page#3 [C000~FFFF]
    end;

  if (iPage = 0) then begin oAddr := (oAddr - 00000); iMapPage := Gi_RAM_PAGE0; end; //Page#0 (0000h~3FFFh)
  if (iPage = 1) then begin oAddr := (oAddr - 16384); iMapPage := Gi_RAM_PAGE1; end; //Page#1 (4000h~7FFFh)
  if (iPage = 2) then begin oAddr := (oAddr - 32768); iMapPage := Gi_RAM_PAGE2; end; //Page#2 (8000h~BFFFh)
  if (iPage = 3) then begin oAddr := (oAddr - 49152); iMapPage := Gi_RAM_PAGE3; end; //Page#3 (C000h~FFFFh)

  Result := MSX_RAM[oAddr + Go_M_Table16384[iMapPage]]; //4000h
end;

procedure PUT_MSX_MEMORY_MAPPER(oAddr: Word; iVal: Byte; iPage: ShortInt);
var
  iMapPage: Byte;
begin
  iMapPage := 0;

  if (iPage = (-1)) then
    begin
      if ((oAddr >= 0) And (oAddr <= 16383)) then iPage := 0; //Page#0 [0000~3FFF]
      if ((oAddr >= 16384) And (oAddr <= 32767)) then iPage := 1; //Page#1 [4000~7FFF]
      if ((oAddr >= 32768) And (oAddr <= 49151)) then iPage := 2; //Page#2 [8000~BFFF]
      if ((oAddr >= 49152) And (oAddr <= 65535)) then iPage := 3; //Page#3 [C000~FFFF]
    end;

  if (iPage = 0) then begin oAddr := (oAddr - 00000); iMapPage := Gi_RAM_PAGE0; end; //Page#0 (0000h~3FFFh)
  if (iPage = 1) then begin oAddr := (oAddr - 16384); iMapPage := Gi_RAM_PAGE1; end; //Page#1 (4000h~7FFFh)
  if (iPage = 2) then begin oAddr := (oAddr - 32768); iMapPage := Gi_RAM_PAGE2; end; //Page#2 (8000h~BFFFh)
  if (iPage = 3) then begin oAddr := (oAddr - 49152); iMapPage := Gi_RAM_PAGE3; end; //Page#3 (C000h~FFFFh)

  MSX_RAM[oAddr + Go_M_Table16384[iMapPage]] := iVal; //4000h
end;

//Konami without SCC (a.k.a.Konami4)
//8Kb - Bank1:4000h~5FFFh, Bank2:6000h~7FFFh, Bank3:8000h~9FFFh, Bank4:A000h~BFFFh
//Chg - Bank1:None, Bank2:6000h~7FFFh(6000h), Bank3:8000h~9FFFh(8000h), Bank4:A000h~BFFFh(A000h)
procedure Move_Rom_To_Page_Konami4(iBank: ShortInt; iPage: Byte);
var
  i: Integer;
  oAddr, oRom: LongInt;
begin
  oAddr := foGetBankAddress8(iBank);
  oRom := foGetPageAddress8(iPage);

  for i := 0 to 8191 do MSX_MEM[1, oAddr + i] := MSX_ROM[oRom + i]; //0000h~1FFFh
end;

//Konami with SCC (a.k.a.Konami5)
//8Kb - Bank1:4000h~5FFFh, Bank2:6000h~7FFFh, Bank3:8000h~9FFFh, Bank4:A000h~BFFFh
//Chg - Bank1:5000h~57FFh(5000h), Bank2:7000h~77FFh(7000h), Bank3:9000h~97FFh(9000h), Bank4:B000h~B7FFh(B000h)
procedure Move_Rom_To_Page_Konami5(iBank: ShortInt; iPage: Byte);
var
  i: Integer;
  oAddr, oRom: LongInt;
begin
  oAddr := foGetBankAddress8(iBank);
  oRom := foGetPageAddress8(iPage);

  for i := 0 to 8191 do MSX_MEM[1, oAddr + i] := MSX_ROM[oRom + i]; //0000h~1FFFh
end;

//ASCII 8Kb
//8Kb - Bank1:4000h~5FFFh, Bank2:6000h~7FFFh, Bank3:8000h~9FFFh, Bank4:A000h~BFFFh
//Chg - Bank1:6000h~67FFh(6000h), Bank2:6800h~6FFFh(6800h), Bank3:7000h~77FFh(7000h), Bank4:7800h~7FFFh(7800h)
procedure Move_Rom_To_Page_Ascii8(iBank: ShortInt; iPage: Byte);
var
  i: Integer;
  oAddr, oRom: LongInt;
begin
  oAddr := foGetBankAddress8(iBank);
  oRom := foGetPageAddress8(iPage);

  for i := 0 to 8191 do MSX_MEM[1, oAddr + i] := MSX_ROM[oRom + i]; //0000h~1FFFh
end;

//ASCII 16Kb
//16Kb - Bank1:4000h~7FFFh, Bank3:8000h~BFFFh
//Chg - Bank1:6000h~67FFh(6000h), Bank3:7000h~77FFh(7000h+77FFh)
procedure Move_Rom_To_Page_Ascii16(iBank: ShortInt; iPage: Byte);
var
  i: Integer;
  oAddr, oRom: LongInt;
begin
  oAddr := foGetBankAddress8(iBank);
  oRom := foGetPageAddress16(iPage);

  for i := 0 to 16383 do MSX_MEM[1, oAddr + i] := MSX_ROM[oRom + i]; //0000h~3FFFh
end;

//64-in-1 and 80-in-1
//8Kb - Bank1:4000h~5FFFh, Bank2:6000h~7FFFh, Bank3:8000h~9FFFh, Bank4:A000h~BFFFh
//Chg - Bank1:(4000h), Bank2:(4001h), Bank3:(4002h), Bank4:(4003h)
procedure Move_Rom_To_Page_80_in_1(iBank: ShortInt; iPage: Byte);
var
  i: Integer;
  oAddr, oRom: LongInt;
begin
  oAddr := foGetBankAddress8(iBank);
  oRom := foGetPageAddress8(iPage);

  for i := 0 to 8191 do MSX_MEM[1, oAddr + i] := MSX_ROM[oRom + i]; //0000h~1FFFh
end;

//126-in-1
//16Kb - Bank1:4000h~7FFFh, Bank3:8000h~BFFFh
//Chg - Bank1:(4000h), Bank3:(4001h)
procedure Move_Rom_To_Page_126_in_1(iBank: ShortInt; iPage: Byte);
var
  i: Integer;
  oAddr, oRom: LongInt;
begin
  oAddr := foGetBankAddress8(iBank);
  oRom := foGetPageAddress16(iPage);

  for i := 0 to 16383 do MSX_MEM[1, oAddr + i] := MSX_ROM[oRom + i]; //0000h~3FFFh
end;

function foGetBankAddress8(iBank: ShortInt): LongInt;
var
  R: LongInt;
begin
  R := 0;
  if (iBank = 1) then R := $4000;
  if (iBank = 2) then R := $6000;
  if (iBank = 3) then R := $8000;
  if (iBank = 4) then R := $A000;

  Result := R;
end;

function foGetPageAddress8(iPage: Byte): LongInt;
begin
  Result := (iPage * 8192); //2000h
end;

function foGetPageAddress16(iPage: Byte): LongInt;
begin
  Result := (iPage * 16384); //4000h
end;

end.

