unit msxZ80A;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Math;

const
  C_Flag = $0100;
  S_Flag16 = $8000;
  S_Flag = $0080;
  H_Flag16 = $1000;
  H_Flag = $0010;

var
  Z80A_REG_PC: Word; //0~65535
  Z80A_REG_SP: Word;

  Z80A_REG_IX: Word;
  Z80A_REG_IY: Word;

  Z80A_REG_A: Byte; //0~255
  Z80A_REG_F: Byte;
  Z80A_REG_B: Byte;
  Z80A_REG_C: Byte;
  Z80A_REG_D: Byte;
  Z80A_REG_E: Byte;
  Z80A_REG_H: Byte;
  Z80A_REG_L: Byte;

  Z80A_REG_A2: Byte;
  Z80A_REG_F2: Byte;
  Z80A_REG_B2: Byte;
  Z80A_REG_C2: Byte;
  Z80A_REG_D2: Byte;
  Z80A_REG_E2: Byte;
  Z80A_REG_H2: Byte;
  Z80A_REG_L2: Byte;

  Z80A_REG_I: Byte;
  Z80A_REG_R: Byte;

  Z80A_REG_IFF1: ShortInt;
  Z80A_REG_IFF2: ShortInt;

  procedure MSX_INIT_Z80A;
  procedure EXEC_Z80A_ADD_PC(iSz: Byte; iSz0: Byte);
  function GET_Z80A_REG_AF: Word;
  function GET_Z80A_REG_BC: Word;
  function GET_Z80A_REG_DE: Word;
  function GET_Z80A_REG_HL: Word;
  function GET_Z80A_REG_AF2: Word;
  function GET_Z80A_REG_BC2: Word;
  function GET_Z80A_REG_DE2: Word;
  function GET_Z80A_REG_HL2: Word;
  procedure PUT_Z80A_REG_AF(oVal: Word);
  procedure PUT_Z80A_REG_BC(oVal: Word);
  procedure PUT_Z80A_REG_DE(oVal: Word);
  procedure PUT_Z80A_REG_HL(oVal: Word);
  procedure PUT_Z80A_REG_AF2(oVal: Word);
  procedure PUT_Z80A_REG_BC2(oVal: Word);
  procedure PUT_Z80A_REG_DE2(oVal: Word);
  procedure PUT_Z80A_REG_HL2(oVal: Word);
  function GET_Z80A_REG_IXH: Byte;
  function GET_Z80A_REG_IXL: Byte;
  function GET_Z80A_REG_IYH: Byte;
  function GET_Z80A_REG_IYL: Byte;
  procedure PUT_Z80A_REG_IXH(iVal: Byte);
  procedure PUT_Z80A_REG_IXL(iVal: Byte);
  procedure PUT_Z80A_REG_IYH(iVal: Byte);
  procedure PUT_Z80A_REG_IYL(iVal: Byte);
  function GET_Z80A_FLAG_S: ShortInt;
  function GET_Z80A_FLAG_Z: ShortInt;
  function GET_Z80A_FLAG_Y: ShortInt;
  function GET_Z80A_FLAG_H: ShortInt;
  function GET_Z80A_FLAG_X: ShortInt;
  function GET_Z80A_FLAG_PV: ShortInt;
  function GET_Z80A_FLAG_N: ShortInt;
  function GET_Z80A_FLAG_C: ShortInt;
  procedure PUT_Z80A_FLAG_S(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_Z(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_Y(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_H(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_X(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_PV(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_N(iVal: ShortInt);
  procedure PUT_Z80A_FLAG_C(iVal: ShortInt);
  function giGetIdx8(iVal: ShortInt): ShortInt;
  function giGet16Sign(oVal: Word): ShortInt;
  function giGet8Sign(iVal: Byte): ShortInt;
  function giGet16Zero(oVal: Word): ShortInt;
  function giGet8Zero(iVal: Byte): ShortInt;
  function goGetLngToLng(oVal: LongInt): Word;
  function giGetIntToInt(iVal: SmallInt): Byte;
  function goGetIntHLToLng(iValH: Byte; iValL: Byte): Word;
  procedure gtMoveLngToIntHL(oVal: Word; var iValH: Byte; var iValL: Byte);
  function giGetLngToIntH(Reg16: Word): Byte;
  function giGetLngToIntL(Reg16: Word): Byte;
  procedure gt_INC_16TO16(oVal: Word; var Reg16: Word);
  procedure gt_DEC_16TO16(oVal: Word; var Reg16: Word);
  procedure gt_INC_16TO8(oVal: Word; var RegH: Byte; var RegL: Byte);
  procedure gt_DEC_16TO8(oVal: Word; var RegH: Byte; var RegL: Byte);
  procedure gt_INC_8TO8(iVal: Byte; var Reg8: Byte);
  procedure gt_DEC_8TO8(iVal: Byte; var Reg8: Byte);
  procedure gt_ADD_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
  procedure gt_SUB_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
  procedure gt_ADD_16TO8(oVal1: Word; oVal2: Word; var RegH: Byte; var RegL: Byte; iCv: ShortInt);
  procedure gt_SUB_16TO8(oVal1: Word; oVal2: Word; var RegH: Byte; var RegL: Byte; iCv: ShortInt);
  procedure ft_ADD_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
  procedure ft_SUB_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
  procedure gt_ADD_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte; iCv: ShortInt);
  procedure gt_SUB_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte; iCv: ShortInt);
  procedure gt_CP(iVal2: Byte);
  procedure gt_AND_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte);
  procedure gt_XOR_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte);
  procedure gt_OR_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte);
  procedure gt_BIT(iBit: ShortInt; Reg8: Byte);
  procedure gt_SET(iBit: ShortInt; var Reg8: Byte);
  procedure gt_RES(iBit: ShortInt; var Reg8: Byte);
  procedure gt_SCF;
  procedure gt_CCF;
  procedure gt_CPL; //비트반전
  procedure gt_NEG; //음수값 : -n, (NOT n) + 1 -> 뺄셈을 음수의 덧셈으로 할 경우
  procedure gt_DAA;
  procedure gt_PUSH_16(oVal: Word);
  procedure gt_POP_16(var Reg16: Word);
  procedure gt_POP_16TO2(var RegH: Byte; var RegL: Byte);
  procedure gt_RLCA;
  procedure gt_RLA;
  procedure gt_RRCA;
  procedure gt_RRA;
  procedure gt_RLC(var Reg8: Byte; iArg: ShortInt);
  procedure gt_RL(var Reg8: Byte; iArg: ShortInt);
  procedure gt_RRC(var Reg8: Byte; iArg: ShortInt);
  procedure gt_RR(var Reg8: Byte; iArg: ShortInt);
  procedure gt_SLA(var Reg8: Byte);
  procedure gt_SLL(var Reg8: Byte);
  procedure gt_SRL(var Reg8: Byte);
  procedure gt_SRA(var Reg8: Byte);
  procedure gt_RLD;
  procedure gt_RRD;
  procedure gt_LDI;
  procedure gt_LDIR;
  procedure gt_LDD;
  procedure gt_LDDR;
  procedure gt_CPI;
  procedure gt_CPIR;
  procedure gt_CPD;
  procedure gt_CPDR;

implementation

uses
  msxMSX, msxMEM, msxTBL;

procedure MSX_INIT_Z80A;
begin
  //
end;

procedure EXEC_Z80A_ADD_PC(iSz: Byte; iSz0: Byte);
begin
  Z80A_REG_PC := ((Z80A_REG_PC + iSz) And $FFFF);
  if (iSz = 0) then iSz := iSz0;
  Gi_Z80A_ClockSize := Gi_Z80A_CLOCK + iSz;
end;

function GET_Z80A_REG_AF: Word;
begin
  Result := ((Z80A_REG_A shl 8) Or Z80A_REG_F);
end;

function GET_Z80A_REG_BC: Word;
begin
  Result := ((Z80A_REG_B shl 8) Or Z80A_REG_C);
end;

function GET_Z80A_REG_DE: Word;
begin
  Result := ((Z80A_REG_D shl 8) Or Z80A_REG_E);
end;

function GET_Z80A_REG_HL: Word;
begin
  Result := ((Z80A_REG_H shl 8) Or Z80A_REG_L);
end;

function GET_Z80A_REG_AF2: Word;
begin
  Result := ((Z80A_REG_A2 shl 8) Or Z80A_REG_F2);
end;

function GET_Z80A_REG_BC2: Word;
begin
  Result := ((Z80A_REG_B2 shl 8) Or Z80A_REG_C2);
end;

function GET_Z80A_REG_DE2: Word;
begin
  Result := ((Z80A_REG_D2 shl 8) Or Z80A_REG_E2);
end;

function GET_Z80A_REG_HL2: Word;
begin
  Result := ((Z80A_REG_H2 shl 8) Or Z80A_REG_L2);
end;

procedure PUT_Z80A_REG_AF(oVal: Word);
begin
  Z80A_REG_A := (oVal shr 8);
  Z80A_REG_F := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_BC(oVal: Word);
begin
  Z80A_REG_B := (oVal shr 8);
  Z80A_REG_C := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_DE(oVal: Word);
begin
  Z80A_REG_D := (oVal shr 8);
  Z80A_REG_E := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_HL(oVal: Word);
begin
  Z80A_REG_H := (oVal shr 8);
  Z80A_REG_L := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_AF2(oVal: Word);
begin
  Z80A_REG_A2 := (oVal shr 8);
  Z80A_REG_F2 := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_BC2(oVal: Word);
begin
  Z80A_REG_B2 := (oVal shr 8);
  Z80A_REG_C2 := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_DE2(oVal: Word);
begin
  Z80A_REG_D2 := (oVal shr 8);
  Z80A_REG_E2 := (oVal And $00FF);
end;

procedure PUT_Z80A_REG_HL2(oVal: Word);
begin
  Z80A_REG_H2 := (oVal shr 8);
  Z80A_REG_L2 := (oVal And $00FF);
end;

function GET_Z80A_REG_IXH: Byte;
begin
  Result := (Z80A_REG_IX shr 8);
end;

function GET_Z80A_REG_IXL: Byte;
begin
  Result := (Z80A_REG_IX And $00FF);
end;

function GET_Z80A_REG_IYH: Byte;
begin
  Result := (Z80A_REG_IY shr 8);
end;

function GET_Z80A_REG_IYL: Byte;
begin
  Result := (Z80A_REG_IY And $00FF);
end;

procedure PUT_Z80A_REG_IXH(iVal: Byte);
begin
  Z80A_REG_IX := ((Z80A_REG_IX And $00FF) Or (iVal shl 8));
end;

procedure PUT_Z80A_REG_IXL(iVal: Byte);
begin
  Z80A_REG_IX := ((Z80A_REG_IX And $FF00) Or (iVal));
end;

procedure PUT_Z80A_REG_IYH(iVal: Byte);
begin
  Z80A_REG_IY := ((Z80A_REG_IY And $00FF) Or (iVal shl 8));
end;

procedure PUT_Z80A_REG_IYL(iVal: Byte);
begin
  Z80A_REG_IY := ((Z80A_REG_IY And $FF00) Or (iVal));
end;

function GET_Z80A_FLAG_S: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 128) = 0, 0, 1); //1000 0000
end;

function GET_Z80A_FLAG_Z: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 64) = 0, 0, 1); //0100 0000
end;

function GET_Z80A_FLAG_Y: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 32) = 0, 0, 1); //0010 0000
end;

function GET_Z80A_FLAG_H: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 16) = 0, 0, 1); //0001 0000
end;

function GET_Z80A_FLAG_X: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 8) = 0, 0, 1); //0000 1000
end;

function GET_Z80A_FLAG_PV: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 4) = 0, 0, 1); //0000 0100
end;

function GET_Z80A_FLAG_N: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 2) = 0, 0, 1); //0000 0010
end;

function GET_Z80A_FLAG_C: ShortInt;
begin
  Result := IfThen((Z80A_REG_F And 1) = 0, 0, 1); //0000 0001
end;

procedure PUT_Z80A_FLAG_S(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $7F); //0111 1111
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 128); //1000 0000
    end;
end;

procedure PUT_Z80A_FLAG_Z(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $BF); //1011 1111
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 64); //0100 0000
    end;
end;

procedure PUT_Z80A_FLAG_Y(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $DF); //1101 1111
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 32); //0010 0000
    end;
end;

procedure PUT_Z80A_FLAG_H(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $EF); //1110 1111
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 16); //0001 0000
    end;
end;

procedure PUT_Z80A_FLAG_X(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $F7); //1111 0111
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 8); //0000 1000
    end;
end;

procedure PUT_Z80A_FLAG_PV(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $FB); //1111 1011
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 4); //0000 0100
    end;
end;

procedure PUT_Z80A_FLAG_N(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $FD); //1111 1101
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 2); //0000 0010
    end;
end;

procedure PUT_Z80A_FLAG_C(iVal: ShortInt);
begin
  if (iVal = 0) then
    begin
      Z80A_REG_F := (Z80A_REG_F And $FE); //1111 1110
    end
  else
    begin
      Z80A_REG_F := (Z80A_REG_F Or 1); //0000 0001
    end;
end;

function giGetIdx8(iVal: ShortInt): ShortInt;
begin
  if (iVal > 127) then
    begin
      Result := (iVal - 256);
    end
  else
    begin
      Result := (iVal);
    end;
end;

function giGet16Sign(oVal: Word): ShortInt;
begin
  //Sign 16 Bit (1:음수,0:양수)
  if (oVal > 32767) then
    begin
      Result := 1;
    end
  else
    begin
      Result := 0;
    end;
end;

function giGet8Sign(iVal: Byte): ShortInt;
begin
  //Sign 8 Bit (1:음수,0:양수)
  if (iVal > 127) then
    begin
      Result := 1;
    end
  else
    begin
      Result := 0;
    end;
end;

function giGet16Zero(oVal: Word): ShortInt;
begin
  //Zero 16 Bit (1:Zero1,0:Other)
  if (oVal = 0) then
    begin
      Result := 1;
    end
  else
    begin
      Result := 0;
    end;
end;

function giGet8Zero(iVal: Byte): ShortInt;
begin
  //Zero 8 Bit (1:Zero1,0:Other)
  if (iVal = 0) then
    begin
      Result := 1;
    end
  else
    begin
      Result := 0;
    end;
end;

function goGetLngToLng(oVal: LongInt): Word;
var
  R: LongInt;
begin
  R := oVal;
  if (R < 0) then R := R + 65536;
  if (R > 65535) then R := R And $FFFF; //-65536;

  Result := R;
end;

function giGetIntToInt(iVal: SmallInt): Byte;
var
  R: SmallInt;
begin
  R := iVal;
  if (R < 0) then R := R + 256;
  if (R > 255) then R := R And $FF; //-256;

  Result := R;
end;

function goGetIntHLToLng(iValH: Byte; iValL: Byte): Word;
begin
  Result := ((iValH shl 8) Or iValL);
end;

procedure gtMoveLngToIntHL(oVal: Word; var iValH: Byte; var iValL: Byte);
begin
  iValH := (oVal shr 8);
  iValL := (oVal And $00FF);
end;

function giGetLngToIntH(Reg16: Word): Byte;
begin
  Result := (Reg16 shr 8);
end;

function giGetLngToIntL(Reg16: Word): Byte;
begin
  Result := (Reg16 And $00FF);
end;

procedure gt_INC_16TO16(oVal: Word; var Reg16: Word);
begin
  //nn=nn+1
  Reg16 := goGetLngToLng(oVal + 1); //INC
end;

procedure gt_DEC_16TO16(oVal: Word; var Reg16: Word);
begin
  //nn=nn-1
  Reg16 := goGetLngToLng(oVal - 1); //DEC
end;

procedure gt_INC_16TO8(oVal: Word; var RegH: Byte; var RegL: Byte);
var
  Reg16: Word;
begin
  //nn=nn+1
  Reg16 := goGetLngToLng(oVal + 1); //INC
  RegH := giGetLngToIntH(Reg16);
  RegL := giGetLngToIntL(Reg16);
end;

procedure gt_DEC_16TO8(oVal: Word; var RegH: Byte; var RegL: Byte);
var
  Reg16: Word;
begin
  //nn=nn-1
  Reg16 := goGetLngToLng(oVal - 1); //DEC
  RegH := giGetLngToIntH(Reg16);
  RegL := giGetLngToIntL(Reg16);
end;

procedure gt_INC_8TO8(iVal: Byte; var Reg8: Byte);
var
  iResult: SmallInt;
begin
  //i=i+1
  //S=Sign(i), Z=Zero(i), H=Hcry(i), Before P=7Fh, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  iResult := (iVal + 1); //INC
  Reg8 := (iResult And $FF);
  PUT_Z80A_FLAG_H(IfThen((iVal And $0F) = 15, 1, 0)); //하프캐리
  PUT_Z80A_FLAG_PV(IfThen(iVal = $7F, 1, 0)); //오버플로
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_DEC_8TO8(iVal: Byte; var Reg8: Byte);
var
  iResult: SmallInt;
begin
  //i=i-1
  //S=Sign(i), Z=Zero(i), H=Hbrw(i), Before P=80h, N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  iResult := (iVal - 1); //DEC
  Reg8 := giGetIntToInt(iResult);
  PUT_Z80A_FLAG_H(IfThen((iVal And $0F) = 0, 1, 0)); //하프캐리
  PUT_Z80A_FLAG_PV(IfThen(iVal = $80, 1, 0)); //오버플로
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_ADD_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
begin
  ft_ADD_16TO16(oVal1, oVal2, Reg16, iCv);
end;

procedure gt_SUB_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
begin
  ft_SUB_16TO16(oVal1, oVal2, Reg16, iCv);
end;

procedure gt_ADD_16TO8(oVal1: Word; oVal2: Word; var RegH: Byte; var RegL: Byte; iCv: ShortInt);
var
  Reg16: Word;
begin
  ft_ADD_16TO16(oVal1, oVal2, Reg16, iCv);
  RegH := giGetLngToIntH(Reg16);
  RegL := giGetLngToIntL(Reg16);
end;

procedure gt_SUB_16TO8(oVal1: Word; oVal2: Word; var RegH: Byte; var RegL: Byte; iCv: ShortInt);
var
  Reg16: Word;
begin
  ft_SUB_16TO16(oVal1, oVal2, Reg16, iCv);
  RegH := giGetLngToIntH(Reg16);
  RegL := giGetLngToIntL(Reg16);
end;

procedure ft_ADD_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
var
  oResult: LongInt; iCarry: ShortInt;
begin
  //nn=nn+rr
  //H=Hcry(nn), N=0, C=Carry(nn) : ADD
  //S=Sign(nn), Z=Zero(nn), P=Oflow(nn) : ADC
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  iCarry := IfThen(iCv = 1, GET_Z80A_FLAG_C, 0); //ADC
  oResult := (oVal1 + oVal2 + iCarry); //ADD
  Reg16 := (oResult And $FFFF);
  PUT_Z80A_FLAG_C(IfThen(Reg16 <> oResult, 1, 0)); //캐리
  PUT_Z80A_FLAG_H(IfThen(((oVal1 And $0FFF) + ((oVal2 + iCarry) And $0FFF)) > $0FFF, 1, 0)); //하프캐리
  if (iCv = 1) then
    begin
      PUT_Z80A_FLAG_PV(IfThen(((oVal1 Xor oResult) And ((oVal2 + iCarry) Xor oResult) And S_Flag16) <> 0, 1, 0)); //오버플로
      PUT_Z80A_FLAG_S(giGet16Sign(Reg16));
      PUT_Z80A_FLAG_Z(giGet16Zero(Reg16));
    end;
end;

procedure ft_SUB_16TO16(oVal1: Word; oVal2: Word; var Reg16: Word; iCv: ShortInt);
var
  oResult: LongInt; iCarry: ShortInt;
begin
  //nn=nn-rr
  //H=Hbrw(nn), N=1, C=Borrow(nn) : SUB
  //S=Sign(nn), Z=Zero(nn), P=Oflow(nn) : SBC
  PUT_Z80A_FLAG_N(1); //뺄셈
  iCarry := IfThen(iCv = 1, GET_Z80A_FLAG_C, 0); //SBC
  oResult := (oVal1 - oVal2 - iCarry); //SUB
  Reg16 := goGetLngToLng(oResult);
  PUT_Z80A_FLAG_C(IfThen(Reg16 <> oResult, 1, 0)); //캐리
  PUT_Z80A_FLAG_H(IfThen((oVal1 And $0FFF) < ((oVal2 + iCarry) And $0FFF), 1, 0)); //하프캐리
  if (iCv = 1) then
    begin
      PUT_Z80A_FLAG_PV(IfThen(((oVal1 Xor (oVal2 + iCarry)) And (oVal1 Xor oResult) And S_Flag16) <> 0, 1, 0)); //오버플로
      PUT_Z80A_FLAG_S(giGet16Sign(Reg16));
      PUT_Z80A_FLAG_Z(giGet16Zero(Reg16));
    end;
end;

procedure gt_ADD_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte; iCv: ShortInt);
var
  iResult: SmallInt; iCarry: ShortInt;
begin
  //i=i+r
  //S=Sign(i), Z=Zero(i), H=Hcry(i), P=Oflow(i), N=0, C=Carry(i)
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  iCarry := IfThen(iCv = 1, GET_Z80A_FLAG_C, 0); //ADC
  iResult := (iVal1 + iVal2 + iCarry); //ADD
  Reg8 := (iResult And $FF);
  PUT_Z80A_FLAG_C(IfThen(Reg8 <> iResult, 1, 0)); //캐리
  PUT_Z80A_FLAG_H(IfThen(((iVal1 And $0F) + ((iVal2 + iCarry) And $0F)) > $0F, 1, 0)); //하프캐리
  PUT_Z80A_FLAG_PV(IfThen(((iVal1 Xor iResult) And ((iVal2 + iCarry) Xor iResult) And S_Flag) <> 0, 1, 0)); //오버플로
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_SUB_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte; iCv: ShortInt);
var
  iResult: SmallInt; iCarry: ShortInt;
begin
  //i=i-r
  //S=Sign(i), Z=Zero(i), H=Hbrw(i), P=Oflow(i), N=0, C=Borrow(i)
  PUT_Z80A_FLAG_N(1); //뺄셈
  iCarry := IfThen(iCv = 1, GET_Z80A_FLAG_C, 0); //SBC
  iResult := (iVal1 - iVal2 - iCarry); //SUB
  Reg8 := giGetIntToInt(iResult);
  PUT_Z80A_FLAG_C(IfThen(Reg8 <> iResult, 1, 0)); //캐리
  PUT_Z80A_FLAG_H(IfThen((iVal1 And $0F) < ((iVal2 + iCarry) And $0F), 1, 0)); //하프캐리
  PUT_Z80A_FLAG_PV(IfThen(((iVal1 Xor (iVal2 + iCarry)) And (iVal1 Xor iResult) And S_Flag) <> 0, 1, 0)); //오버플로
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_CP(iVal2: Byte);
var
  iResult: SmallInt; Reg8: Byte;
begin
  //CP=i-r
  //S=Sign(CP), Z=Zero(CP), H=Hbrw(CP), P=Oflow(CP), N=1, C=Borrow(CP)
  PUT_Z80A_FLAG_N(1); //뺄셈
  iResult := (Z80A_REG_A - iVal2); //SUB
  Reg8 := giGetIntToInt(iResult);
  PUT_Z80A_FLAG_C(IfThen(Reg8 <> iResult, 1, 0)); //캐리
  PUT_Z80A_FLAG_H(IfThen((Z80A_REG_A And $0F) < (iVal2 And $0F), 1, 0)); //하프캐리
  PUT_Z80A_FLAG_PV(0); //오버플로
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_AND_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte);
begin
  //i=i And r
  //S=Sign(i), Z=Zero(i), H=1, P=Peven(i), N=0, C=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  Reg8 := (iVal1 And iVal2); //AND
  PUT_Z80A_FLAG_C(0); //Not캐리
  PUT_Z80A_FLAG_H(1); //하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_XOR_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte);
begin
  //i=i Xor r
  //S=Sign(i), Z=Zero(i), H=0, P=Peven(i), N=0, C=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  Reg8 := (iVal1 Xor iVal2); //XOR
  PUT_Z80A_FLAG_C(0); //Not캐리
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_OR_8TO8(iVal1: Byte; iVal2: Byte; var Reg8: Byte);
begin
  //i=i Or r
  //S=Sign(i), Z=Zero(i), H=0, P=Peven(i), N=0, C=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  Reg8 := (iVal1 Or iVal2); //OR
  PUT_Z80A_FLAG_C(0); //Not캐리
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_BIT(iBit: ShortInt; Reg8: Byte);
begin
  //Z=bit=0, H=1, N=0
  PUT_Z80A_FLAG_N(0); //뺄셈
  PUT_Z80A_FLAG_Z(IfThen((Reg8 And Gi_B_Table[iBit]) = 0, 1, 0)); //1:Zero
  PUT_Z80A_FLAG_H(1); //하프캐리
end;

procedure gt_SET(iBit: ShortInt; var Reg8: Byte);
begin
  //bit<=1
  Reg8 := (Reg8 Or Gi_B_Table[iBit]); //1
end;

procedure gt_RES(iBit: ShortInt; var Reg8: Byte);
begin
  //bit<=0
  Reg8 := (Reg8 And Gi_NB_Table[iBit]); //0
end;

procedure gt_SCF;
begin
  //C=1, H=0, N=0
  PUT_Z80A_FLAG_N(0); //NOT제로
  PUT_Z80A_FLAG_H(0); //하프캐리
  PUT_Z80A_FLAG_C(1); //캐리
end;

procedure gt_CCF;
var
  C: ShortInt;
begin
  //H=C, N=0, C=Not(C)
  PUT_Z80A_FLAG_N(0); //NOT제로
  C := GET_Z80A_FLAG_C;
  PUT_Z80A_FLAG_H(C); //하프캐리
  if (C = 0) then //캐리
    begin
      PUT_Z80A_FLAG_C(1);
    end
  else
    begin
      PUT_Z80A_FLAG_C(0);
    end;
end;

procedure gt_CPL; //비트반전
begin
  //A=Not(A), H=1, N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  PUT_Z80A_FLAG_H(1); //하프캐리
  Z80A_REG_A := giGetIntToInt(Not Z80A_REG_A);
end;

procedure gt_NEG; //음수값 : -n, (NOT n) + 1 -> 뺄셈을 음수의 덧셈으로 할 경우
var
  iRegOrg: Byte; iResult: SmallInt;
begin
  //P=Before A=80h, C=Before A<>00h, A=0-A, S=Sigh(A), Z=Zero(A), H=Hcr(A), N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  iRegOrg := Z80A_REG_A;
  iResult := (0 - iRegOrg);
  Z80A_REG_A := giGetIntToInt(iResult);
  PUT_Z80A_FLAG_C(IfThen(iRegOrg <> 0, 1, 0)); //캐리
  PUT_Z80A_FLAG_H(IfThen(((iRegOrg Xor iResult) And H_Flag) <> 0, 1, 0)); //하프캐리
  PUT_Z80A_FLAG_PV(IfThen(iRegOrg = $80, 1, 0)); //오버플로
  PUT_Z80A_FLAG_S(giGet8Sign(Z80A_REG_A)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Z80A_REG_A)); //Zero
end;

procedure gt_DAA;
var
  iRegA: Byte; iAdj: Byte; iResult: SmallInt; iResCry: ShortInt; iH: Byte; iL: Byte; F_H: ShortInt;
begin
  //DAA
  //S=Sign(결과), Z=Zero(결과), H=하프캐리(결과), P=Peven(결과), C=작업에따라변경
  //Debug.Print " N:" + CStr(Z80A_FLAG_N) + " A:" + Right("00" + Hex(Z80A_REG_A), 2) + " C:" + CStr(Z80A_FLAG_C) + " H:" + CStr(Z80A_FLAG_H)
  iRegA := Z80A_REG_A;
  iAdj := 0;
  iResult := 0;
  iResCry := 0;
  iH := (iRegA shr 4);
  iL := (iRegA And $0F);
  F_H := GET_Z80A_FLAG_H;
  if (GET_Z80A_FLAG_N = 0) then //ADD
    begin
      if (GET_Z80A_FLAG_C = 0) then
        begin
          if ((iH <= $09) And (F_H = 0) And (iL <= $09)) then begin iAdj := $00; end;
          if ((iH <= $08) And (F_H = 0) And (iL >= $0A)) then begin iAdj := $06; end;
          if ((iH <= $09) And (F_H = 1) And (iL <= $03)) then begin iAdj := $06; end;
          if ((iH >= $0A) And (F_H = 0) And (iL <= $09)) then begin iAdj := $60; iResCry := 1; end;
          if ((iH >= $09) And (F_H = 0) And (iL >= $0A)) then begin iAdj := $66; iResCry := 1; end;
          if ((iH >= $0A) And (F_H = 1) And (iL <= $03)) then begin iAdj := $66; iResCry := 1; end;
        end
      else
        begin
          if ((iH <= $02) And (F_H = 0) And (iL <= $09)) then begin iAdj := $60; iResCry := 1; end;
          if ((iH <= $02) And (F_H = 0) And (iL >= $0A)) then begin iAdj := $66; iResCry := 1; end;
          if ((iH <= $03) And (F_H = 1) And (iL <= $03)) then begin iAdj := $66; iResCry := 1; end;
        end;
    end
  else //SUB
    begin
      if (GET_Z80A_FLAG_C = 0) then
        begin
          if ((iH <= $09) And (F_H = 0) And (iL <= $09)) then begin iAdj := $00; end;
          if ((iH <= $08) And (F_H = 1) And (iL >= $06)) then begin iAdj := $FA; end;
        end
      else
        begin
          if ((iH >= $07) And (F_H = 0) And (iL <= $09)) then begin iAdj := $A0; iResCry := 1; end;
          if ((iH >= $06) And (F_H = 1) And (iL >= $06)) then begin iAdj := $9A; iResCry := 1; end;
        end;
    end;
    iResult := (iRegA + iAdj);
    Z80A_REG_A := giGetIntToInt(iResult);
    PUT_Z80A_FLAG_H(IfThen(((iRegA And $F) + (iAdj And $F)) > $F, 1, 0)); //하프캐리
    PUT_Z80A_FLAG_C(iResCry); //캐리
    PUT_Z80A_FLAG_PV(Gi_P_Table[Z80A_REG_A]); //1:짝수,0:홀수
    PUT_Z80A_FLAG_S((giGet8Sign(Z80A_REG_A))); //Sign
    PUT_Z80A_FLAG_Z((giGet8Zero(Z80A_REG_A))); //Zero
end;

procedure gt_PUSH_16(oVal: Word);
begin
  PUT_MSX_MEMORY(goGetLngToLng(Z80A_REG_SP - 1), giGetLngToIntH(oVal));
  PUT_MSX_MEMORY(goGetLngToLng(Z80A_REG_SP - 2), giGetLngToIntL(oVal));
  Z80A_REG_SP := goGetLngToLng(Z80A_REG_SP - 2);
end;

procedure gt_POP_16(var Reg16: Word);
begin
  Reg16 := goGetIntHLToLng(GET_MSX_MEMORY(((Z80A_REG_SP + 1) And $FFFF)), GET_MSX_MEMORY(Z80A_REG_SP));
  Z80A_REG_SP := ((Z80A_REG_SP + 2) And $FFFF);
end;

procedure gt_POP_16TO2(var RegH: Byte; var RegL: Byte);
begin
  RegH := GET_MSX_MEMORY(((Z80A_REG_SP + 1) And $FFFF));
  RegL := GET_MSX_MEMORY(Z80A_REG_SP);
  Z80A_REG_SP := ((Z80A_REG_SP + 2) And $FFFF);
end;

procedure gt_RLCA;
begin
  gt_RLC(Z80A_REG_A, 1);
end;

procedure gt_RLA;
begin
  gt_RL(Z80A_REG_A, 1);
end;

procedure gt_RRCA;
begin
  gt_RRC(Z80A_REG_A, 1);
end;

procedure gt_RRA;
begin
  gt_RR(Z80A_REG_A, 1);
end;

procedure gt_RLC(var Reg8: Byte; iArg: ShortInt);
var
  C: ShortInt;
begin
  //H=0, N=0, C=1[1]
  //P=Peven, S=Sign, Z=Zero
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := IfThen((Reg8 And 128) = 0, 0, 1); //1[1]
  Reg8 := (((Reg8 shl 1) And 255) + C); //2[7]+C
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  if (iArg = 0) then
   begin
     PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
     PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
     PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
   end;
end;

procedure gt_RL(var Reg8: Byte; iArg: ShortInt);
var
  C: ShortInt;
begin
  //H=0, N=0, C=1[1]
  //P=Peven, S=Sign, Z=Zero
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := IfThen((Reg8 And 128) = 0, 0, 1); //1[1]
  Reg8 := (((Reg8 shl 1) And 255) + GET_Z80A_FLAG_C); //2[7]+FC
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  if (iArg = 0) then
    begin
      PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
      PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
      PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
    end;
end;

procedure gt_RRC(var Reg8: Byte; iArg: ShortInt);
var
  C: ShortInt;
begin
  //H=0, N=0, C=8[1]
  //P=Peven, S=Sign, Z=Zero
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := (Reg8 And 1); //8[1]
  Reg8 := ((C shl 7) + (Reg8 shr 1)); //C+1[7]
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  if (iArg = 0) then
    begin
      PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
      PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
      PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
    end;
end;

procedure gt_RR(var Reg8: Byte; iArg: ShortInt);
var
  C: ShortInt;
begin
  //H=0, N=0, C=8[1]
  //P=Peven, S=Sign, Z=Zero
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := (Reg8 And 1); //8[1]
  Reg8 := ((GET_Z80A_FLAG_C shl 7) + (Reg8 shr 1)); //FC+1[7]
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  if (iArg = 0) then
    begin
      PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
      PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
      PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
    end;
end;

procedure gt_SLA(var Reg8: Byte);
var
  C: ShortInt;
begin
  //S=Sign, Z=Zero, H=0, P=Peven, N=0, C=1[1]
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := IfThen((Reg8 And 128) = 0, 0, 1); //1[1]
  Reg8 := (((Reg8 shl 1) And 255)); //2[7]+0
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_SLL(var Reg8: Byte);
var
  C: ShortInt;
begin
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := IfThen((Reg8 And 128) = 0, 0, 1); //1[1]
  Reg8 := (((Reg8 shl 1) And 255) + 1); //2[7]+1 //+1버그명령???
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_SRL(var Reg8: Byte);
var
  C: ShortInt;
begin
  //S=0, Z=Zero, H=0, P=Peven, N=0, C=8[1]
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := (Reg8 And 1); //8[1]
  Reg8 := (Reg8 shr 1); //0+1[7]
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(0); //giGet8Sign(Reg8) //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_SRA(var Reg8: Byte);
var
  C, C1: ShortInt;
begin
  //S=Sign, Z=Zero, H=0, P=Peven, N=0, C=8[1]
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  C := (Reg8 And 1); //8[1]
  C1 := IfThen((Reg8 And 128) = 0, 0, 1); //1[1]
  Reg8 := ((C1 shl 7) + (Reg8 shr 1)); //1[1]+1[7]
  PUT_Z80A_FLAG_C(C);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_RLD;
var
 Reg8, iVal, iValx, iValA1, iValA2, iValB1, iValB2: Byte;
begin
  //A,(HL)
  //S=Sign, Z=Zero, H=0, P=Peven, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  iValA1 := ((Z80A_REG_A shr 4));
  iValA2 := (Z80A_REG_A And 15);
  iVal := GET_MSX_MEMORY(GET_Z80A_REG_HL);
  iValB1 := (iVal shr 4);
  iValB2 := (iVal And 15);
  iValx := iValA2;
  iValA2 := iValB1;
  iValB1 := iValB2;
  iValB2 := iValx;
  Reg8 := ((iValA1 shl 4) + iValA2);
  Z80A_REG_A := Reg8;
  PUT_MSX_MEMORY(GET_Z80A_REG_HL, ((iValB1 shl 4) + iValB2));
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_RRD;
var
 Reg8, iVal, iValx, iValA1, iValA2, iValB1, iValB2: Byte;
begin
  //A,(HL)
  //S=Sign, Z=Zero, H=0, P=Peven, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  iValA1 := (Z80A_REG_A shr 4);
  iValA2 := (Z80A_REG_A And 15);
  iVal := GET_MSX_MEMORY(GET_Z80A_REG_HL);
  iValB1 := (iVal shr 4);
  iValB2 := (iVal And 15);
  iValx := iValA2;
  iValA2 := iValB2;
  iValB2 := iValB1;
  iValB1 := iValx;
  Reg8 := ((iValA1 shl 4) + iValA2);
  Z80A_REG_A := Reg8;
  PUT_MSX_MEMORY(GET_Z80A_REG_HL, ((iValB1 shl 4) + iValB2));
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(Gi_P_Table[Reg8]); //1:짝수,0:홀수
  PUT_Z80A_FLAG_S(giGet8Sign(Reg8)); //Sign
  PUT_Z80A_FLAG_Z(giGet8Zero(Reg8)); //Zero
end;

procedure gt_LDI;
begin
  //(DE)=(HL), HL=HL+1, DE=DE+1, BC=BC-1
  //H=0, P=BC-1<>0, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  PUT_MSX_MEMORY(GET_Z80A_REG_DE, GET_MSX_MEMORY(GET_Z80A_REG_HL));
  PUT_Z80A_REG_HL(goGetLngToLng(GET_Z80A_REG_HL + 1));
  PUT_Z80A_REG_DE(goGetLngToLng(GET_Z80A_REG_DE + 1));
  PUT_Z80A_REG_BC(goGetLngToLng(GET_Z80A_REG_BC - 1));
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(IfThen(GET_Z80A_REG_BC > 0, 1, 0)); //오버플로
end;

procedure gt_LDIR;
var
  HL, DE, BC: Word;
label
  LoopProc;
begin
  //Repeat (DE)=(HL), HL=HL+1, DE=DE+1, BC=BC-1 While (BC<>0)
  //H=0, P=BC-1<>0, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  HL := GET_Z80A_REG_HL;
  DE := GET_Z80A_REG_DE;
  BC := GET_Z80A_REG_BC;
LoopProc: ;
  PUT_MSX_MEMORY(DE, GET_MSX_MEMORY(HL));
  HL := goGetLngToLng(HL + 1);
  DE := goGetLngToLng(DE + 1);
  BC := goGetLngToLng(BC - 1);
  if (BC > 0) then
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 21;
      goto LoopProc;
    end
  else
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 16;
    end;
  PUT_Z80A_REG_HL(HL);
  PUT_Z80A_REG_DE(DE);
  PUT_Z80A_REG_BC(BC);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(0); //Not오버플로
end;

procedure gt_LDD;
begin
  //(DE)=(HL), HL=HL-1, DE=DE-1, BC=BC-1
  //H=0, P=BC-1<>0, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  PUT_MSX_MEMORY(GET_Z80A_REG_DE, GET_MSX_MEMORY(GET_Z80A_REG_HL));
  PUT_Z80A_REG_HL(goGetLngToLng(GET_Z80A_REG_HL - 1));
  PUT_Z80A_REG_DE(goGetLngToLng(GET_Z80A_REG_DE - 1));
  PUT_Z80A_REG_BC(goGetLngToLng(GET_Z80A_REG_BC - 1));
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(IfThen(GET_Z80A_REG_BC > 0, 1, 0)); //오버플로
end;

procedure gt_LDDR;
var
  HL, DE, BC: Word;
label
  LoopProc;
begin
  //Repeat (DE)=(HL), HL=HL-1, DE=DE-1, BC=BC-1 While (BC<>0)
  //H=0, P=BC-1<>0, N=0
  PUT_Z80A_FLAG_N(0); //Not뺄셈
  HL := GET_Z80A_REG_HL;
  DE := GET_Z80A_REG_DE;
  BC := GET_Z80A_REG_BC;
LoopProc: ;
  PUT_MSX_MEMORY(DE, GET_MSX_MEMORY(HL));
  HL := goGetLngToLng(HL - 1);
  DE := goGetLngToLng(DE - 1);
  BC := goGetLngToLng(BC - 1);
  if (BC > 0) then
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 21;
      goto LoopProc;
    end
  else
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 16;
    end;
  PUT_Z80A_REG_HL(HL);
  PUT_Z80A_REG_DE(DE);
  PUT_Z80A_REG_BC(BC);
  PUT_Z80A_FLAG_H(0); //Not하프캐리
  PUT_Z80A_FLAG_PV(0); //Not오버플로
end;

procedure gt_CPI;
begin
  //CP=A-(HL), HL=HL+1, BC=BC-1
  //S=Sign(CP), Z=Zero(CP), H=Hcry(CP), PV=BC-1<>0, N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  gt_CP(GET_MSX_MEMORY(GET_Z80A_REG_HL));
  PUT_Z80A_REG_HL(goGetLngToLng(GET_Z80A_REG_HL + 1));
  PUT_Z80A_REG_BC(goGetLngToLng(GET_Z80A_REG_BC - 1));
  //Z80A_FLAG_C := ? //캐리
  //Z80A_FLAG_H := ? //하프캐리 -> CP결과
  //Z80A_FLAG_S := ? //Sign     -> CP결과
  //Z80A_FLAG_Z := ? //Zero     -> CP결과
  PUT_Z80A_FLAG_PV(IfThen(GET_Z80A_REG_BC > 0, 1, 0)); //오버플로
end;

procedure gt_CPIR;
var
  HL, BC: Word;
label
  LoopProc;
begin
  //Repeat CP=A-(HL), HL=HL+1, BC=BC-1 While (BC<>0) And (Z<>1)
  //S=Sign(CP), Z=Zero(CP), H=Hcry(CP), PV=BC-1<>0, N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  HL := GET_Z80A_REG_HL;
  BC := GET_Z80A_REG_BC;
LoopProc: ;
  gt_CP(GET_MSX_MEMORY(HL));
  HL := goGetLngToLng(HL + 1);
  BC := goGetLngToLng(BC - 1);
  if ((BC <> 0) And (GET_Z80A_FLAG_Z <> 1)) then
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 21;
      goto LoopProc;
    end
  else
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 16;
    end;
  PUT_Z80A_REG_HL(HL);
  PUT_Z80A_REG_BC(BC);
  //Z80A_FLAG_C := ? //캐리
  //Z80A_FLAG_H := ? //하프캐리 -> CP결과
  //Z80A_FLAG_S := ? //Sign     -> CP결과
  //Z80A_FLAG_Z := ? //Zero     -> CP결과
  PUT_Z80A_FLAG_PV(IfThen(GET_Z80A_REG_BC > 0, 1, 0)); //오버플로
end;

procedure gt_CPD;
begin
  //CP=A-(HL), HL=HL-1, BC=BC-1
  //S=Sign(CP), Z=Zero(CP), H=Hcry(CP), PV=BC-1<>0, N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  gt_CP(GET_MSX_MEMORY(GET_Z80A_REG_HL));
  PUT_Z80A_REG_HL(goGetLngToLng(GET_Z80A_REG_HL - 1));
  PUT_Z80A_REG_BC(goGetLngToLng(GET_Z80A_REG_BC - 1));
  //Z80A_FLAG_C := ? //캐리
  //Z80A_FLAG_H := ? //하프캐리 -> CP결과
  //Z80A_FLAG_S := ? //Sign     -> CP결과
  //Z80A_FLAG_Z := ? //Zero     -> CP결과
  PUT_Z80A_FLAG_PV(IfThen(GET_Z80A_REG_BC > 0, 1, 0)); //오버플로
end;

procedure gt_CPDR;
var
  HL, BC: Word;
label
  LoopProc;
begin
  //Repeat CP=A-(HL), HL=HL-1, BC=BC-1 While (BC<>0) And (Z<>1)
  //S=Sign(CP), Z=Zero(CP), H=Hcry(CP), PV=BC-1<>0, N=1
  PUT_Z80A_FLAG_N(1); //뺄셈
  HL := GET_Z80A_REG_HL;
  BC := GET_Z80A_REG_BC;
LoopProc: ;
  gt_CP(GET_MSX_MEMORY(HL));
  HL := goGetLngToLng(HL - 1);
  BC := goGetLngToLng(BC - 1);
  if ((BC <> 0) And (GET_Z80A_FLAG_Z <> 1)) then
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 21;
      goto LoopProc;
    end
  else
    begin
      Gi_Z80A_CLOCK := Gi_Z80A_CLOCK + 16;
    end;
  PUT_Z80A_REG_HL(HL);
  PUT_Z80A_REG_BC(BC);
  //Z80A_FLAG_C := ? //캐리
  //Z80A_FLAG_H := ? //하프캐리 -> CP결과
  //Z80A_FLAG_S := ? //Sign     -> CP결과
  //Z80A_FLAG_Z := ? //Zero     -> CP결과
  PUT_Z80A_FLAG_PV(IfThen(GET_Z80A_REG_BC > 0, 1, 0)); //오버플로
end;

end.

